#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched/signal.h>
#include <linux/rcupdate.h>

/*
 * Show function: prints the page table operation counters for every active process.
 * Each line follows the format:
 * [pid]: [[pgd_alloc,pgd_free,pgd_set]], [[pud_alloc,pud_free,pud_set]],
 *        [[pmd_alloc,pmd_free,pmd_set]], [[pte_alloc,pte_free,pte_set]]
 */
static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    rcu_read_lock();
    for_each_process(task) {
        seq_printf(m, "%d: [[%lu,%lu,%lu]], [[%lu,%lu,%lu]], [[%lu,%lu,%lu]], [[%lu,%lu,%lu]]\n",
                   task->pid,
                   task->pgd_alloc, task->pgd_free, task->pgd_set,
                   task->pud_alloc, task->pud_free, task->pud_set,
                   task->pmd_alloc, task->pmd_free, task->pmd_set,
                   task->pte_alloc, task->pte_free, task->pte_set);
    }
    rcu_read_unlock();
    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

/*
 * Use the new procfs API with struct proc_ops.
 */
static const struct proc_ops pg_stats_ops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

static int __init pg_stats_init(void)
{
    /* Create the /proc/pg_stats entry using the new API */
    proc_create("pg_stats", 0, NULL, &pg_stats_ops);
    return 0;
}
module_init(pg_stats_init);

static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
}
module_exit(pg_stats_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Your Name");
MODULE_DESCRIPTION("Memory Management Statistics");
