#include <linux/init.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/seq_file.h>
#include <linux/rcupdate.h>


MODULE_LICENSE("GPL");
MODULE_AUTHOR("s2157894");
MODULE_DESCRIPTION("Module to show page table stats in /proc/pg_stats");

#define ENTRY_NAME "pg_stats"
#define PERMS 0644
#define PARENT NULL


static void *ct_seq_start(struct seq_file *s, loff_t *pos)
{

    rcu_read_lock();
    //*pos = 0; // Start at position 0 on each start
    struct task_struct *task;
    /*
        for (p = &init_task ; (p = next_task(p)) != &init_task ; )
    */
    if (*pos == 0){
        task = &init_task;
        task = next_task(task);
        return task;
    }else{
        *pos = 0;
        return NULL;
    } 
}

static void *ct_seq_next(struct seq_file *s, void *v, loff_t *pos)
{     

    struct task_struct *task = v;


    (*pos)++; // Needs to be at start of this function (before we return anything), otherwise has angry error message

    
    /*
    p = next_task(p))
    */
    task = next_task(task);

    /*!= &init_task*/
    if (task == &init_task){
        return NULL;
    }


    return task;
}

static void ct_seq_stop(struct seq_file *s, void *v)
{   

    rcu_read_unlock();
}

static int ct_seq_show(struct seq_file *s, void *v)
{   
    struct task_struct *task = v;


    // Validate the task pointer
    if (task == &init_task) { // Skip if back to init_task, this will be handled in next function
        return 0;
    }

    /*Skip kernel threads - according to sched.h, we can use EXIT_TRACE to check for zombie threads
    */
    if (!task->mm){
        return 0;
    }

    /* A bit messy, not sure if there is an intended way to format it like this, output looks the same though */
    seq_printf(s, "[%d]: [[%u],[%u],[%u]], [[%u],[%u],[%u]], [[%u],[%u],[%u]], [[%u],[%u],[%u]]\n", task->pid,
                task->pgd_allocate_count, task->pgd_free_count, task->pgd_set_count,
                task->pud_allocate_count, task->pud_free_count, task->pud_set_count,
                task->pmd_allocate_count, task->pmd_free_count, task->pmd_set_count,
                task->pte_allocate_count, task->pte_free_count, task->pte_set_count
            );
    return 0;
}

static const struct seq_operations ct_seq_ops = {
    .start = ct_seq_start,
    .next  = ct_seq_next,
    .stop  = ct_seq_stop,
    .show  = ct_seq_show
};

static int ct_open(struct inode *inode, struct file *file)
{
    return seq_open(file, &ct_seq_ops);
}


static const struct proc_ops ct_file_ops = {
    .proc_open = ct_open,
    .proc_read_iter = seq_read_iter, 
    .proc_lseek = seq_lseek,   
    .proc_release = seq_release, 
};

static int __init pgstats_init(void)
{
    printk(KERN_INFO "/proc/%s created\n", ENTRY_NAME);

    proc_create(ENTRY_NAME, PERMS, NULL, &ct_file_ops);
    return 0;
}

static void __exit pgstats_exit(void)
{
    remove_proc_entry(ENTRY_NAME, NULL);

    printk(KERN_INFO "Removing /proc/%s.\n", ENTRY_NAME);
}

module_init(pgstats_init);
module_exit(pgstats_exit);