#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h> // for for_each_process()
#include <linux/mm.h>

static int pg_stats_show(struct seq_file *m, void *v)
{
    /*
     * We'll iterate over every *active* process in the system and print
     * its page table counters in the required format:
     *
     *   [pid]: [[pgd_alloc],[pgd_free],[pgd_set]],
     *           [[pud_alloc],[pud_free],[pud_set]],
     *           [[pmd_alloc],[pmd_free],[pmd_set]],
     *           [[pte_alloc],[pte_free],[pte_set]]
     */
    struct task_struct *p;

    // Iterate over all processes
    for_each_process(p) {
        if (p->pid == 0) {
            // Skip the swapper (pid 0)
            continue;
        }
        seq_printf(m, "%d: [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], "
                      "[[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]]\n",
                   p->pid,
                   p->pgd_alloc_count, p->pgd_free_count, p->pgd_set_count,
                   p->pud_alloc_count, p->pud_free_count, p->pud_set_count,
                   p->pmd_alloc_count, p->pmd_free_count, p->pmd_set_count,
                   p->pte_alloc_count, p->pte_free_count, p->pte_set_count);
    }

    return 0;
}

/*
 * 'open' callback for /proc/pg_stats, using the seq_file interface.
 */
static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

/*
 * Define proc_ops for /proc/pg_stats.
 * Note: Newer kernel versions require proc_ops rather than file_operations.
 */
static const struct proc_ops pg_stats_proc_ops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

/*
 * Module init: create the /proc/pg_stats entry
 */
static int __init pg_stats_init(void)
{
    proc_create("pg_stats", 0, NULL, &pg_stats_proc_ops);
    pr_info("pg_stats: /proc/pg_stats created.\n");
    return 0;
}

/*
 * Module exit: remove the /proc/pg_stats entry
 */
static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
    pr_info("pg_stats: /proc/pg_stats removed.\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("chris");
MODULE_DESCRIPTION("Displays per-process page-table counters");
