#ifdef CONFIG_PG_STATS
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched/signal.h>

#define PROC_NAME "pg_stats"

// **Function to output process details to seq_file**
static int show_processes(struct seq_file *m, void *v) {
    struct task_struct *task;

    rcu_read_lock();  // Protect task traversal
    for_each_process(task) {
        if (!task->mm)  // Avoid NULL pointer dereference
            continue;
        seq_printf(m, "%d: ", task->pid);
        seq_printf(m, "[[%d],[%d],[%d]], ",
            task->pgd_alloc, 
            task->pgd_free,
            task->pgd_set);
        seq_printf(m, "[[%d],[%d],[%d]], ",
            task->pud_alloc, 
            task->pud_free,
            task->pud_set);
        seq_printf(m, "[[%d],[%d],[%d]], ",
            task->pmd_alloc, 
            task->pmd_free,
            task->pmd_set);
        seq_printf(m, "[[%d],[%d],[%d]]\n",
            task->pte_alloc, 
            task->pte_free,
            task->pte_set);
    }
    rcu_read_unlock();
    
    return 0;
}

// **Open function for seq_file**
static int pg_stats_open(struct inode *inode, struct file *file) { 
    return single_open(file, show_processes, NULL);
}

// **Define file operations**
static const struct proc_ops proc_fops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

// **Initialize module (create proc file)**
static int __init proc_init(void) {
    proc_create(PROC_NAME, 0, NULL, &proc_fops);
    printk(KERN_INFO "/proc/%s created\n", PROC_NAME);
    return 0;
}

// **Cleanup module (remove proc file)**
static void __exit proc_cleanup(void) {
    remove_proc_entry(PROC_NAME, NULL);
    printk(KERN_INFO "/proc/%s removed\n", PROC_NAME);
}

module_init(proc_init);
module_exit(proc_cleanup);

#endif  // CONFIG_PG_STATS