#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/module.h>
#include <linux/init.h>

// structure inspired from stat.c
// and https://devarea.com/linux-kernel-development-creating-a-proc-file-and-interfacing-with-user-space/
// https://embetronicx.com/tutorials/linux/device-drivers/procfs-in-linux/
static int pg_stats_show(struct seq_file *m, void *v) {
    struct task_struct *task;

    for_each_process(task) {
        seq_printf(m, "[%d]: [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]], [[%lu],[%lu],[%lu]]\n",
            task->pid,
            task->pgd_allocate_count, task->pgd_free_count, task->pgd_set_count,
            task->pud_allocate_count, task->pud_free_count, task->pud_set_count,
            task->pmd_allocate_count, task->pmd_free_count, task->pmd_set_count,
            task->pte_allocate_count, task->pte_free_count, task->pte_set_count
        );
    }
    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file) {
    return single_open(file, pg_stats_show, NULL);
}

static const struct proc_ops pg_stats_file_operations = {
    .proc_open = pg_stats_open,
    .proc_read = seq_read,
    .proc_lseek = seq_lseek,
    .proc_release = single_release,
};

static int __init pg_stats_init(void) {
    proc_create("pg_stats", 0, NULL, &pg_stats_file_operations);
    printk(KERN_INFO "pg_stats module loaded\n");
    return 0;
}

static void __exit pg_stats_exit(void) {
    printk(KERN_INFO "pg_stats module unloaded\n");
    remove_proc_entry("pg_stats", NULL);
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("CW2");
MODULE_DESCRIPTION("CW2 TASK 1: Proc file system entry for tracking memory stats");
