#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/sched/signal.h>

static int pg_stats_show(struct seq_file *m, void *v)
{
    struct task_struct *task;

    rcu_read_lock();
    for_each_process(task) {
        seq_printf(m,
                   "[%d]: [[%llu],[%llu],[%llu]], "
                   "[[%llu],[%llu],[%llu]], "
                   "[[%llu],[%llu],[%llu]], "
                   "[[%llu],[%llu],[%llu]]\n",
                    task->pid,
                    atomic64_read(&task->pgd_alloc_count), atomic64_read(&task->pgd_free_count), atomic64_read(&task->pgd_set_count),
                    atomic64_read(&task->pud_alloc_count), atomic64_read(&task->pud_free_count), atomic64_read(&task->pud_set_count),
                    atomic64_read(&task->pmd_alloc_count), atomic64_read(&task->pmd_free_count), atomic64_read(&task->pmd_set_count),
                    atomic64_read(&task->pte_alloc_count), atomic64_read(&task->pte_free_count), atomic64_read(&task->pte_set_count)
                   );
    }
    rcu_read_unlock();

    return 0;
}

static int pg_stats_open(struct inode *inode, struct file *file)
{
    return single_open(file, pg_stats_show, NULL);
}

/*
 * Define a proc_ops instead of file_operations
 * for use with proc_create().
 */
static const struct proc_ops pg_stats_ops = {
    .proc_open    = pg_stats_open,
    .proc_read    = seq_read,
    .proc_lseek   = seq_lseek,
    .proc_release = single_release,
};

static int __init pg_stats_init(void)
{
    /*
     * Use &pg_stats_ops instead of &pg_stats_fops:
     */
    proc_create("pg_stats", 0, NULL, &pg_stats_ops);
    pr_info("pg_stats module loaded\n");
    return 0;
}

static void __exit pg_stats_exit(void)
{
    remove_proc_entry("pg_stats", NULL);
    pr_info("pg_stats module unloaded\n");
}

module_init(pg_stats_init);
module_exit(pg_stats_exit);
