// SPDX-License-Identifier: GPL-2.0
#include <linux/fs.h>
#include <linux/xattr.h>
#include <linux/slab.h>
#include <linux/uio.h>
#include <linux/module.h>
#include <linux/namei.h>

static ssize_t xor_encrypted_read_iter(struct kiocb *iocb, struct iov_iter *to)
{
    struct file *file = iocb->ki_filp;
    const struct file_operations *orig_fops = file->f_op;
    struct dentry *dentry = file->f_path.dentry;
    struct mnt_idmap *idmap = file->f_path.mnt->mnt_idmap;
    ssize_t ret;
    u8 key;
    void *buf;
    size_t len = iov_iter_count(to);

    // Get the encryption key from xattr
    ret = vfs_getxattr(idmap, dentry, "user.cw3.encrypt", (void *)&key, sizeof(key));
    if (ret != sizeof(key)) {
        // Fall back to original read if no valid xattr
        return orig_fops->read_iter(iocb, to);
    }

    // Allocate temporary buffer
    buf = kmalloc(len, GFP_KERNEL);
    if (!buf)
        return -ENOMEM;

    // Create a temporary iterator for reading into our buffer
    struct iov_iter tmp_iter;
    iov_iter_kvec(&tmp_iter, ITER_DEST, (const struct kvec[]){{ buf, len }}, 1, len);

    // Use original read_iter to get the data
    ret = orig_fops->read_iter(iocb, &tmp_iter);
    
    if (ret > 0) {
        // XOR the data with the key
        for (ssize_t i = 0; i < ret; i++) {
            ((u8 *)buf)[i] ^= key;
        }
        
        // Copy to userspace
        if (copy_to_iter(buf, ret, to) != ret)
            ret = -EFAULT;
    }

    kfree(buf);
    return ret;
}

static const struct file_operations xor_encrypted_fops = {
    .read_iter = xor_encrypted_read_iter,
    // Other operations will fall back to the original ones
};

void setup_xor_encryption(struct file *file)
{
    struct dentry *dentry = file->f_path.dentry;
    struct mnt_idmap *idmap = file->f_path.mnt->mnt_idmap;
    u8 key;
    int ret;

    ret = vfs_getxattr(idmap, dentry, "user.cw3.encrypt", (void *)&key, sizeof(key));
    
    if (ret == sizeof(key) && key >= 0 && key <= 255) {
        // Save original file operations
        file->private_data = (void *)file->f_op;
        // Replace with our operations
        file->f_op = &xor_encrypted_fops;
    }
}
EXPORT_SYMBOL(setup_xor_encryption);

static int __init xor_crypt_init(void)
{
    pr_info("XOR encryption module loaded\n");
    return 0;
}

static void __exit xor_crypt_exit(void)
{
    pr_info("XOR encryption module unloaded\n");
}

module_init(xor_crypt_init);
module_exit(xor_crypt_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Your Name");
MODULE_DESCRIPTION("XOR encryption on read for files with user.cw3.encrypt xattr");