/*
 * VMotiONOS - Process Thread Migration System
 * 
 * Main implementation for thread data extraction and transmission
 */

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/pid.h>
#include <linux/pid_namespace.h>
#include <linux/sched.h>
#include <linux/sched/task_stack.h>
#include <linux/sched/mm.h>
#include <linux/sched/signal.h>
#include <linux/mm.h>
#include <linux/mm_types.h>
#include <linux/mmap_lock.h>
#include <linux/fs.h>
#include <linux/file.h>
#include <linux/fdtable.h>
#include <linux/dcache.h>
#include <linux/path.h>
#include <linux/types.h>
#include <linux/mman.h>
#include <linux/maple_tree.h>
#include <linux/ptrace.h>
#include <linux/printk.h>
#include <linux/seq_file.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <asm/ptrace.h>
#include <asm/processor.h>
#include <linux/errno.h>
#include <linux/sched/signal.h>
#include <linux/freezer.h>
#include <linux/ptrace.h>


/* Ensure PIDTYPE_PID is available */
#ifndef PIDTYPE_PID
#define PIDTYPE_PID 0
#endif

#include "vmotionos.h"
#include "vmotionos_server.h"
#include "vmotionos_client.h"

#define PROC_VMOTIONS "vmotionos"
#define MAX_BUFFER_SIZE 256

#define SUSPEND_TIMEOUT_MS 1000
#define SUSPEND_POLL_INTERVAL_MS 10

/* Global variables for proc interface */
static struct proc_dir_entry *proc_vmotions_entry;
static char proc_buffer[MAX_BUFFER_SIZE];
static unsigned long proc_buffer_size = 0;
static int vmotionos_suspend_thread(struct task_struct *task, pid_t pid);

/**
 * Extract thread data from a process
 */
int vmotionos_extract_thread_data(pid_t pid, struct vmotionos_thread *thread)
{
    struct task_struct *task;
    struct mm_struct *mm;
    struct vm_area_struct *vma;
    struct ma_state mas;
    struct pt_regs *source_regs;
    struct thread_struct *source_thread;
    int vma_count = 0;
    int ret = 0;
    
    if (!thread)
        return -EINVAL;
    
    /* Initialize thread structure */
    memset(thread, 0, sizeof(struct vmotionos_thread));
    
    /* Find the task structure */
    rcu_read_lock();
    task = pid_task(find_vpid(pid), PIDTYPE_PID);
    if (!task) {
        rcu_read_unlock();
        vmotionos_err("Process with PID %d not found\n", pid);
        return -ESRCH;
    }

    /* Release RCU lock before suspending (msleep() calls schedule()) */
    rcu_read_unlock();

    /* Suspend the THREAD ONLY (not entire process) */
    ret = vmotionos_suspend_thread(task, pid);
    if (ret < 0) {
        vmotionos_err("Failed to suspend thread PID %d: %d\n", pid, ret);
        return ret;
    }
    
    /* Re-acquire RCU lock for accessing task data */
    rcu_read_lock();
    
    /* Get and check mm */
    mm = task->mm;
    if (!mm) {
        rcu_read_unlock();
        vmotionos_err("Process PID %d has no mm_struct\n", pid);
        return -EINVAL;
    }
    
    /* Extract basic process information */
    thread->pid = pid;
    strncpy(thread->comm, task->comm, MAX_COMM_LEN - 1);
    thread->comm[MAX_COMM_LEN - 1] = '\0';
    
    /* Extract CPU register state */
    source_regs = task_pt_regs(task);
    source_thread = &task->thread;
    
    if (source_regs) {
        /* Copy all general purpose and special registers */
        thread->regs.r15 = source_regs->r15;
        thread->regs.r14 = source_regs->r14;
        thread->regs.r13 = source_regs->r13;
        thread->regs.r12 = source_regs->r12;
        thread->regs.bp = source_regs->bp;
        thread->regs.bx = source_regs->bx;
        thread->regs.r11 = source_regs->r11;
        thread->regs.r10 = source_regs->r10;
        thread->regs.r9 = source_regs->r9;
        thread->regs.r8 = source_regs->r8;
        thread->regs.ax = source_regs->ax;
        thread->regs.cx = source_regs->cx;
        thread->regs.dx = source_regs->dx;
        thread->regs.si = source_regs->si;
        thread->regs.di = source_regs->di;
        thread->regs.ip = source_regs->ip;
        thread->regs.sp = source_regs->sp;
        thread->regs.flags = source_regs->flags;
        thread->regs.orig_ax = source_regs->orig_ax;
        thread->regs.cs = source_regs->cs;
        thread->regs.ss = source_regs->ss;
        
        /* Extract FS/GS bases from thread structure */
        thread->regs.fsbase = source_thread->fsbase;
        thread->regs.gsbase = source_thread->gsbase;
        
        vmotionos_debug("Captured CPU registers - RIP: 0x%lx, RSP: 0x%lx, RAX: 0x%lx",
                        thread->regs.ip, thread->regs.sp, thread->regs.ax);
    } else {
        vmotionos_warn("Failed to get pt_regs, zeroing register state");
        memset(&thread->regs, 0, sizeof(thread->regs));
    }
    
    /* Extract memory management data from actual mm_struct */
    thread->mm.start_code = mm->start_code;
    thread->mm.end_code = mm->end_code;
    thread->mm.start_data = mm->start_data;
    thread->mm.end_data = mm->end_data;
    thread->mm.start_brk = mm->start_brk;
    thread->mm.brk = mm->brk;
    thread->mm.start_stack = mm->start_stack;
    thread->mm.arg_start = mm->arg_start;
    thread->mm.arg_end = mm->arg_end;
    thread->mm.env_start = mm->env_start;
    thread->mm.env_end = mm->env_end;
    
    vmotionos_debug("Extracted real memory layout: code=0x%lx-0x%lx, data=0x%lx-0x%lx, stack=0x%lx",
                    mm->start_code, mm->end_code, mm->start_data, mm->end_data, mm->start_stack);
    
    /* Extract VMA information */
    vmotionos_debug("Extracting VMAs for process '%s' (PID %d)", thread->comm, pid);
    
    /* Initialize VMA iterator and extract VMA data */
    mas_init(&mas, &mm->mm_mt, 0);
    mas_for_each(&mas, vma, ULONG_MAX) {
        char *path_buf;
        char *path;
        
        if (vma_count >= MAX_VMA_COUNT) {
            vmotionos_warn("Too many VMAs, truncating at %d\n", MAX_VMA_COUNT);
            break;
        }
        
        thread->mm.vmas[vma_count].vm_start = vma->vm_start;
        thread->mm.vmas[vma_count].vm_end = vma->vm_end;
        thread->mm.vmas[vma_count].vm_flags = vma->vm_flags;
        thread->mm.vmas[vma_count].vm_pgoff = vma->vm_pgoff;
        thread->mm.vmas[vma_count].vm_page_prot = pgprot_val(vma->vm_page_prot);
        
        /* Get file path if VMA is backed by a file */
        if (vma->vm_file) {
            path_buf = kmalloc(MAX_PATH_LEN, GFP_KERNEL);
            if (path_buf) {
                path = d_path(&vma->vm_file->f_path, path_buf, MAX_PATH_LEN);
                if (!IS_ERR(path)) {
                    strncpy(thread->mm.vmas[vma_count].vm_file_path, path, MAX_PATH_LEN - 1);
                    thread->mm.vmas[vma_count].vm_file_path[MAX_PATH_LEN - 1] = '\0';
                    thread->mm.vmas[vma_count].has_file = 1;
                } else {
                    strcpy(thread->mm.vmas[vma_count].vm_file_path, "[unknown]");
                    thread->mm.vmas[vma_count].has_file = 1;
                }
                kfree(path_buf);
            } else {
                strcpy(thread->mm.vmas[vma_count].vm_file_path, "[no_mem]");
                thread->mm.vmas[vma_count].has_file = 1;
            }
        } else {
            thread->mm.vmas[vma_count].vm_file_path[0] = '\0';
            thread->mm.vmas[vma_count].has_file = 0;
        }
        
        vma_count++;
    }
    
    thread->mm.vma_count = vma_count;
    rcu_read_unlock();
    
    vmotionos_info("Successfully extracted thread data for '%s' (PID %d) with %d VMAs\n", 
                   thread->comm, pid, vma_count);
    
    /* Extract file descriptor information */
    ret = vmotionos_extract_files_data(task, &thread->files);
    if (ret < 0) {
        vmotionos_warn("Failed to extract files data: %d\n", ret);
    }

    /* THREAD REMAINS SUSPENDED - No automatic resume! */
    vmotionos_info("Thread PID %d remains suspended for migration (no resume)\n", pid);
    
    return ret;
}






/**
 * Suspend thread using freezer API - THREAD SPECIFIC ONLY!
 */
static int vmotionos_suspend_thread(struct task_struct *task, pid_t pid)
{
	unsigned long timeout = 100; /* 1 second timeout */
	
	vmotionos_info("=== FREEZER API: Suspending ONLY thread PID %d ===\n", pid);
	vmotionos_info("This will NOT affect other threads in the process!\n");
	
	/* Check task state before freezing */
	vmotionos_info("Task PID %d state before freezing:\n", pid);
	vmotionos_info("  Task state: %u (%s)\n", task->__state,
		       task->__state == TASK_RUNNING ? "RUNNING" :
		       task->__state == TASK_INTERRUPTIBLE ? "INTERRUPTIBLE" :
		       task->__state == TASK_UNINTERRUPTIBLE ? "UNINTERRUPTIBLE" :
		       task->__state == TASK_STOPPED ? "STOPPED" : "OTHER");
	vmotionos_info("  Task flags: 0x%x\n", task->flags);
	vmotionos_info("  Is kernel thread: %s\n", (task->flags & PF_KTHREAD) ? "YES" : "NO");
	vmotionos_info("  Already frozen: %s\n", frozen(task) ? "YES" : "NO");
	
	/* Check if task is already frozen */
	if (frozen(task)) {
		vmotionos_info("Thread PID %d is already frozen\n", pid);
		return 0;
	}
	
	/* Check if task is freezable */
	if (task->flags & PF_KTHREAD) {
		vmotionos_warn("Thread PID %d is a kernel thread - may not be freezable\n", pid);
	}
	
	/* THE KERNEL FREEZER WON'T WORK - It's designed for system-wide suspend only!
	 * From kernel docs: freeze_task() only works when freezer_active static key is set
	 * during system suspend/hibernation. For individual thread suspension, we need ptrace.
	 */
	 
	vmotionos_info("Using ptrace-based thread suspension (INDIVIDUAL thread only)...\n");
	
	/* Method 1: Use kernel ptrace mechanism for thread-specific suspension */
	/* This is what debuggers like GDB use to stop individual threads */
	
	/* Set the thread to traced state - this stops it from running */
	task->ptrace |= PT_PTRACED;
	task->__state = TASK_TRACED;
	
	/* Send SIGSTOP specifically to this thread (not process-wide) */
	/* Use send_sig_info with thread-specific targeting */
	send_sig_info(SIGSTOP, SEND_SIG_PRIV, task);
	
	/* Wait for the thread to actually stop */
	timeout = 100; /* 1 second timeout */
	while (task->__state != TASK_TRACED && task->__state != TASK_STOPPED && timeout > 0) {
		msleep(10);
		timeout--;
		
		/* Check current state */
		if (task->__state == TASK_TRACED) {
			vmotionos_info("Thread entered TASK_TRACED state\n");
			break;
		} else if (task->__state == TASK_STOPPED) {
			vmotionos_info("Thread entered TASK_STOPPED state\n");
			break;
		}
	}
	
	if (task->__state == TASK_TRACED || task->__state == TASK_STOPPED) {
		vmotionos_info("Thread PID %d successfully suspended using ptrace approach (siblings unaffected)\n", pid);
		vmotionos_info("Final thread state: %u (%s)\n", task->__state,
			       task->__state == TASK_TRACED ? "TRACED" : "STOPPED");
		return 0;
	} else {
		vmotionos_err("Ptrace-based suspension failed for PID %d (state: %u)\n", pid, task->__state);
		return -EAGAIN;
	}
}





/**
 * Extract file descriptor table data from task
 */
int vmotionos_extract_files_data(struct task_struct *task, struct vmotionos_files *files)
{
    struct files_struct *task_files;
    struct fdtable *fdt;
    int fd_count = 0;
    int max_check;
    int fd;
    
    if (!task || !files) return -EINVAL;
    
    /* Initialize files structure */
    memset(files, 0, sizeof(struct vmotionos_files));
    
    vmotionos_debug("Extracting file descriptors for task PID %d\n", task->pid);
    
    task_lock(task);
    task_files = task->files;
    if (!task_files) {
        task_unlock(task);
        vmotionos_warn("Task has no files_struct");
        return 0;
    }
    
    /* Do all work while holding task_lock to ensure consistency */
    spin_lock(&task_files->file_lock);
    fdt = files_fdtable(task_files);
    
    /* Copy file table metadata */
    files->next_fd = task_files->next_fd;
    files->max_fds = fdt->max_fds;
    files->resize_in_progress = task_files->resize_in_progress;
    
    /* Extract individual file descriptors */
    max_check = min_t(unsigned int, fdt->max_fds, MAX_FD_COUNT);
    
    for (fd = 0; fd < max_check; fd++) {
        if (test_bit(fd, fdt->open_fds)) {
            struct file *file = fdt->fd[fd];
            char *path_buf;
            char *path;
            
            if (file && fd_count < MAX_FD_COUNT) {
                files->fds[fd_count].fd_num = fd;
                files->fds[fd_count].f_flags = file->f_flags;
                files->fds[fd_count].f_pos = file->f_pos;
                files->fds[fd_count].is_valid = 1;
                files->fds[fd_count].close_on_exec = test_bit(fd, fdt->close_on_exec);
                
                /* Try to get file path */
                path_buf = kmalloc(MAX_PATH_LEN, GFP_ATOMIC);
                if (path_buf) {
                    path = d_path(&file->f_path, path_buf, MAX_PATH_LEN);
                    if (!IS_ERR(path)) {
                        strncpy(files->fds[fd_count].f_path, path, MAX_PATH_LEN - 1);
                        files->fds[fd_count].f_path[MAX_PATH_LEN - 1] = '\0';
                    } else {
                        snprintf(files->fds[fd_count].f_path, MAX_PATH_LEN, "[fd:%d]", fd);
                    }
                    kfree(path_buf);
                } else {
                    snprintf(files->fds[fd_count].f_path, MAX_PATH_LEN, "[fd:%d]", fd);
                }
                
                fd_count++;
            }
        }
    }
    
    files->fd_count = fd_count;
    
    spin_unlock(&task_files->file_lock);
    task_unlock(task);
    
    vmotionos_info("Successfully extracted %d file descriptors\n", fd_count);
    
    /* Print extracted FDs for debugging */
#if VMOTIONOS_DEBUG
    int i;
    if (fd_count > 0) {
        for (i = 0; i < fd_count; i++) {
            vmotionos_debug("FD[%d]: num=%d, flags=0x%x, path='%s'\n",
                           i, files->fds[i].fd_num, files->fds[i].f_flags, files->fds[i].f_path);
        }
    }
#endif
    
    return 0;
}

/**
 * Print files data for verification
 */
void vmotionos_print_files_data(struct vmotionos_files *files)
{
    int i;
    
    if (!files) return;
    
    vmotionos_info("File Descriptor Table:");
    vmotionos_info("  next_fd: %u, max_fds: %u, resize_in_progress: %s\n",
                   files->next_fd, files->max_fds, 
                   files->resize_in_progress ? "true" : "false");
    vmotionos_info("  fd_count: %d\n", files->fd_count);
    
    for (i = 0; i < files->fd_count; i++) {
        struct vmotionos_fd *fd = &files->fds[i];
        if (fd->is_valid) {
            vmotionos_info("  FD %d: flags=0x%x, pos=%lu, close_on_exec=%s, path='%s'\n",
                          fd->fd_num, fd->f_flags, fd->f_pos,
                          fd->close_on_exec ? "true" : "false",
                          fd->f_path);
        }
    }
}

/**
 * Determine VMA type based on flags, file path, and memory regions
 */
static const char* vmotionos_get_vma_type(struct vmotionos_vma *vma, struct vmotionos_mm *mm)
{
    /* Define VM constants if not available */
    #ifndef VM_EXEC
    #define VM_EXEC         0x00000004
    #endif
    #ifndef VM_GROWSDOWN
    #define VM_GROWSDOWN    0x00000100
    #endif
    #ifndef VM_HUGETLB
    #define VM_HUGETLB      0x00400000
    #endif
    #ifndef VM_PFNMAP
    #define VM_PFNMAP       0x00000400
    #endif
    #ifndef VM_IO
    #define VM_IO           0x00004000
    #endif
    #ifndef VM_SHARED
    #define VM_SHARED       0x00000008
    #endif
    #ifndef VM_DONTEXPAND
    #define VM_DONTEXPAND   0x00040000
    #endif
    
    /* Check for special file-backed VMAs first */
    if (vma->has_file) {
        /* Check for VDSO */
        if (strstr(vma->vm_file_path, "[vdso]") || strstr(vma->vm_file_path, "vdso")) {
            return "VDSO";
        }
        /* Check for VVAR (VDSO variables) */
        if (strstr(vma->vm_file_path, "[vvar]")) {
            return "VVAR";
        }
        /* Check for VSYSCALL */
        if (strstr(vma->vm_file_path, "[vsyscall]")) {
            return "VSYSCALL";
        }
        /* Check for shared libraries */
        if (strstr(vma->vm_file_path, ".so") || strstr(vma->vm_file_path, "/lib")) {
            if (vma->vm_flags & VM_EXEC) {
                return "LIB_EXEC";
            } else {
                return "LIB_DATA";
            }
        }
        /* Check for main executable */
        if (vma->vm_flags & VM_EXEC) {
            return "EXEC";
        } else {
            return "FILE_DATA";
        }
    }
    
    /* Special detection for VDSO/VVAR based on characteristics */
    /* Check for PFNMAP regions in high memory address space */
    if ((vma->vm_flags & VM_PFNMAP) && vma->vm_start > 0x7f0000000000UL) {
        /* Debug info for VDSO/VVAR detection */
        vmotionos_debug("High memory PFNMAP VMA detected: 0x%lx-0x%lx, flags=0x%lx\n", 
                       vma->vm_start, vma->vm_end, vma->vm_flags);
        vmotionos_debug("  VM_PFNMAP: %s, VM_IO: %s, VM_DONTEXPAND: %s, VM_EXEC: %s\n",
                       (vma->vm_flags & VM_PFNMAP) ? "YES" : "NO",
                       (vma->vm_flags & VM_IO) ? "YES" : "NO", 
                       (vma->vm_flags & VM_DONTEXPAND) ? "YES" : "NO",
                       (vma->vm_flags & VM_EXEC) ? "YES" : "NO");
        
        /* Check if it has typical VDSO characteristics (small size, high address) */
        unsigned long size = vma->vm_end - vma->vm_start;
        if (size <= (16 * PAGE_SIZE)) { /* VDSO/VVAR are typically small */
            /* Check for typical VDSO/VVAR flag combinations */
            if (vma->vm_flags & (VM_PFNMAP | VM_IO | VM_DONTEXPAND)) {
                /* VDSO is typically executable */
                if (vma->vm_flags & VM_EXEC) {
                    return "VDSO";
                }
                /* VVAR is typically read-only with special flags */
                else if ((vma->vm_flags & VM_EXEC) == 0) {
                    return "VVAR";
                }
            }
        }
        
        /* Fallback for other PFNMAP regions in high memory */
        return "PFNMAP_HIGH";
    }
    
    /* Anonymous VMAs - determine type based on location and flags */
    if (vma->vm_flags & VM_GROWSDOWN) {
        return "STACK";
    }
    
    /* Check memory regions */
    if (vma->vm_start <= mm->start_brk && vma->vm_end >= mm->brk) {
        return "HEAP";
    }
    if (vma->vm_start >= mm->start_code && vma->vm_end <= mm->end_code) {
        return "CODE";
    }
    if (vma->vm_start >= mm->start_data && vma->vm_end <= mm->end_data) {
        return "DATA";
    }
    
    /* Check special flags */
    if (vma->vm_flags & VM_HUGETLB) {
        return "HUGETLB";
    }
    if (vma->vm_flags & VM_PFNMAP) {
        /* Could be device mapping or other special memory */
        if (vma->vm_flags & VM_IO) {
            return "IO_PFNMAP";
        } else {
            return "PFNMAP";
        }
    }
    if (vma->vm_flags & VM_IO) {
        return "IO";
    }
    if (vma->vm_flags & VM_SHARED) {
        return "ANON_SHARED";
    }
    
    return "ANON_PRIVATE";
}

/**
 * Print thread data for verification - SHOWS ALL VMAs
 */
void vmotionos_print_thread_data(struct vmotionos_thread *thread)
{
    int i;
    
    if (!thread) return;
    
    vmotionos_info("=== VMotionOS Thread Data ===");
    vmotionos_info("Process: %s (PID: %d)\n", thread->comm, thread->pid);
    
    vmotionos_info("CPU Registers:");
    vmotionos_info("  RIP: 0x%016lx  RSP: 0x%016lx  RBP: 0x%016lx\n", 
                   thread->regs.ip, thread->regs.sp, thread->regs.bp);
    vmotionos_info("  RAX: 0x%016lx  RBX: 0x%016lx  RCX: 0x%016lx\n", 
                   thread->regs.ax, thread->regs.bx, thread->regs.cx);
    vmotionos_info("  FS_BASE: 0x%016lx  GS_BASE: 0x%016lx\n", 
                   thread->regs.fsbase, thread->regs.gsbase);
    
    /* Print ALL VMAs (not limited to first 3) */
    vmotionos_info("=== ALL VMA INFORMATION (%d VMAs) ===\n", thread->mm.vma_count);
    for (i = 0; i < thread->mm.vma_count; i++) {
        const char *vma_type = vmotionos_get_vma_type(&thread->mm.vmas[i], &thread->mm);
        vmotionos_info("  VMA[%d]: 0x%016lx-0x%016lx flags=0x%lx [%s] %s\n", 
                       i, thread->mm.vmas[i].vm_start, thread->mm.vmas[i].vm_end, 
                       thread->mm.vmas[i].vm_flags, vma_type,
                       thread->mm.vmas[i].has_file ? thread->mm.vmas[i].vm_file_path : "[anon]");
    }
    vmotionos_info("=== END VMA INFORMATION (%d total VMAs) ===\n", thread->mm.vma_count);
    
    /* Print file descriptor information */
    vmotionos_print_files_data(&thread->files);
    
    vmotionos_info("=============================");
}

/**
 * Send thread data to remote server
 */
int vmotionos_send_thread_data(struct vmotionos_thread *thread, char *dest_ip, int dest_port)
{
    struct vmotionos_message *typed_msg;
    char *message_buffer;
    size_t total_msg_size, basic_size, regs_size, mm_meta_size, files_meta_size, vma_size, fd_size;
    char *write_ptr;
    struct vmotionos_message_params msg;
    int ret;
    
    if (!thread || !dest_ip) return -EINVAL;
    
    /* Component-by-component packing to preserve structure layout */
    basic_size = sizeof(int) + MAX_COMM_LEN;
    regs_size = sizeof(struct vmotionos_regs);
    mm_meta_size = sizeof(struct vmotionos_mm) - (MAX_VMA_COUNT * sizeof(struct vmotionos_vma));
    files_meta_size = sizeof(struct vmotionos_files) - (MAX_FD_COUNT * sizeof(struct vmotionos_fd));
    vma_size = thread->mm.vma_count * sizeof(struct vmotionos_vma);  
    fd_size = thread->files.fd_count * sizeof(struct vmotionos_fd);
    
    total_msg_size = sizeof(struct vmotionos_message) + basic_size + regs_size + 
                     mm_meta_size + files_meta_size + vma_size + fd_size;
    
    vmotionos_info("Sending %zu bytes (%d VMAs, %d FDs)\n",
                   total_msg_size, thread->mm.vma_count, thread->files.fd_count);
    
    vmotionos_debug("Message breakdown: basic=%zu + regs=%zu + mm=%zu + files=%zu + vmas=%zu + fds=%zu\n",
                    basic_size, regs_size, mm_meta_size, files_meta_size, vma_size, fd_size);
    
    /* Create typed message */
    message_buffer = kzalloc(total_msg_size, GFP_KERNEL);
    if (!message_buffer) {
        vmotionos_err("Failed to allocate message buffer (%zu bytes)\n", total_msg_size);
        return -ENOMEM;
    }
    
    typed_msg = (struct vmotionos_message *)message_buffer;
    typed_msg->msg_type = MSG_TYPE_CREATE_THREAD;
    typed_msg->data_size = total_msg_size - sizeof(struct vmotionos_message);
    
    /* Pack components sequentially */
    write_ptr = typed_msg->data;
    
    /* 1. Copy basic info: pid + comm */
    vmotionos_debug("Packing PID: %d (0x%x)\n", thread->pid, thread->pid);
    memcpy(write_ptr, &thread->pid, sizeof(int));
    write_ptr += sizeof(int);
    memcpy(write_ptr, thread->comm, MAX_COMM_LEN);
    write_ptr += MAX_COMM_LEN;
    
    /* 2. Copy CPU registers */
    memcpy(write_ptr, &thread->regs, regs_size);
    write_ptr += regs_size;
    
    /* 3. Copy MM metadata */
    memcpy(write_ptr, &thread->mm, mm_meta_size);
    write_ptr += mm_meta_size;
    
    /* 4. Copy files metadata */  
    memcpy(write_ptr, &thread->files, files_meta_size);
    write_ptr += files_meta_size;
    
    /* 5. Copy actual VMAs */
    if (vma_size > 0) {
        memcpy(write_ptr, thread->mm.vmas, vma_size);
        write_ptr += vma_size;
        vmotionos_debug("Copied %d VMAs (%zu bytes)\n", thread->mm.vma_count, vma_size);
    }
    
    /* 6. Copy actual FDs */
    if (fd_size > 0) {
        memcpy(write_ptr, thread->files.fds, fd_size);
        write_ptr += fd_size;
        vmotionos_debug("Copied %d FDs (%zu bytes)\n", thread->files.fd_count, fd_size);
    }
    
    /* Send UDP message */
    msg.dest_ip = dest_ip;
    msg.dest_port = dest_port;
    msg.message = message_buffer;
    msg.msg_len = total_msg_size;
    
    ret = vmotionos_send_message(&msg);
    if (ret < 0) {
        vmotionos_err("Failed to send CREATE_THREAD message: %d\n", ret);
    } else {
        vmotionos_info("Successfully sent CREATE_THREAD message for PID %d\n", thread->pid);
    }
    
    kfree(message_buffer);
    return ret;
}

/**
 * Handle read operations on /proc/vmotions
 */
#ifndef ssize_t
typedef long ssize_t;
#endif
#ifndef loff_t  
typedef long long loff_t;
#endif

static ssize_t vmotions_proc_read(struct file *file, char __user *user_buffer, 
                                 size_t count, loff_t *pos)
{
    if (*pos >= proc_buffer_size) {
        return 0; /* EOF */
    }
    
    if (count > proc_buffer_size - *pos) {
        count = proc_buffer_size - *pos;
    }
    
    return simple_read_from_buffer(user_buffer, count, pos, proc_buffer, proc_buffer_size);
}

/**
 * Handle write operations on /proc/vmotions
 */
static ssize_t vmotions_proc_write(struct file *file, const char __user *user_buffer,
                                  size_t count, loff_t *pos)
{
    char *local_buffer;
    int pid_value;
    struct vmotionos_thread *thread;
    int ret;
    
    if (count > MAX_BUFFER_SIZE - 1) {
        vmotionos_warn("Write count %zu exceeds maximum buffer size %d\n", 
                       count, MAX_BUFFER_SIZE - 1);
        return -EINVAL;
    }
    
    local_buffer = kzalloc(count + 1, GFP_KERNEL);
    if (!local_buffer) {
        vmotionos_err("Failed to allocate memory for write buffer\n");
        return -ENOMEM;
    }
    
    if (copy_from_user(local_buffer, user_buffer, count)) {
        vmotionos_err("Failed to copy data from user space\n");
        kfree(local_buffer);
        return -EFAULT;
    }
    
    local_buffer[count] = '\0';
    
    /* Parse the PID value */
    ret = kstrtoint(local_buffer, 10, &pid_value);
    if (ret) {
        vmotionos_err("Invalid PID format: '%s' (error: %d)\n", local_buffer, ret);
        kfree(local_buffer);
        return -EINVAL;
    }
    
    vmotionos_info("Received PID parameter: %d\n", pid_value);
    
    /* Allocate thread structure */
    thread = kzalloc(sizeof(struct vmotionos_thread), GFP_KERNEL);
    if (!thread) {
        vmotionos_err("Failed to allocate thread structure\n");
        kfree(local_buffer);
        return -ENOMEM;
    }
    
    /* Extract thread data */
    ret = vmotionos_extract_thread_data(pid_value, thread);
    if (ret < 0) {
        vmotionos_err("Failed to extract thread data: %d\n", ret);
        kfree(thread);
        kfree(local_buffer);
        return ret;
    }
    
    /* Print thread data for verification (shows ALL VMAs now) */
    vmotionos_print_thread_data(thread);
    
    /* Send thread data to server */
    ret = vmotionos_send_thread_data(thread, "192.168.122.81", SERVER_PORT);
    if (ret < 0) {
        vmotionos_err("Failed to send thread data: %d\n", ret);
    }
    
    /* Update proc buffer */
    proc_buffer_size = snprintf(proc_buffer, MAX_BUFFER_SIZE, 
                               "Last PID processed: %d (%s)\nSent CREATE_THREAD message with %d VMAs, %d FDs\n", 
                               thread->pid, thread->comm, thread->mm.vma_count, thread->files.fd_count);
    
    kfree(thread);
    kfree(local_buffer);
    return count;
}

/* Proc file operations */
static const struct proc_ops vmotions_proc_ops = {
    .proc_read_iter = NULL,
    .proc_write = vmotions_proc_write,
    .proc_read = vmotions_proc_read,
};

/**
 * Initialize the VMotionOS system
 */
static int __init vmotionos_init(void)
{
    int ret;

    vmotionos_info("Initializing system...\n");

    /* Create /proc/vmotions entry */
    proc_vmotions_entry = proc_create(PROC_VMOTIONS, 0666, NULL, &vmotions_proc_ops);
    if (!proc_vmotions_entry) {
        vmotionos_err("Failed to create /proc/%s entry\n", PROC_VMOTIONS);
        return -ENOMEM;
    }
    
    vmotionos_info("/proc/%s created successfully\n", PROC_VMOTIONS);

    /* Initialize proc buffer */
    proc_buffer_size = snprintf(proc_buffer, MAX_BUFFER_SIZE, 
                               "VMotionOS ready\nWrite PID to extract and send thread data\n");

    /* Start the UDP server */
    ret = vmotionos_server_start();
    if (ret < 0) {
        vmotionos_err("Failed to start server, error %d\n", ret);
        proc_remove(proc_vmotions_entry);
        return ret;
    }

    vmotionos_info("System initialized successfully\n");
    vmotionos_info("Server running on port %d\n", SERVER_PORT);
    vmotionos_info("Usage: echo PID > /proc/%s\n", PROC_VMOTIONS);

    return 0;
}

static void __exit vmotionos_exit(void)
{
    vmotionos_info("Exiting system...\n");
    
    /* Stop server */
    vmotionos_server_stop();
    
    /* Remove proc entry */
    if (proc_vmotions_entry) {
        proc_remove(proc_vmotions_entry);
        vmotionos_info("/proc/%s removed\n", PROC_VMOTIONS);
    }
    
    /* Cleanup client */
    vmotionos_client_cleanup();
}

late_initcall(vmotionos_init);
module_exit(vmotionos_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("VMotionOS Team");
MODULE_DESCRIPTION("Process Thread Migration System");
MODULE_VERSION("1.0");
