/*
 * VMotiONOS - Process Thread Migration System
 * 
 * Clean header for core functionality
 */

#ifndef _VMOTIONOS_H
#define _VMOTIONOS_H

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/printk.h>

/* Configuration constants */
#define MAX_VMA_COUNT 256  
#define MAX_PATH_LEN 256   
#define MAX_COMM_LEN 16    
#define MAX_FD_COUNT 64

/* Message types for network communication */
#define MSG_TYPE_CREATE_THREAD 1
#define MSG_TYPE_DESTROY_THREAD 2
#define MSG_TYPE_MIGRATE_THREAD 3
#define MSG_TYPE_PAGE_REQUEST 4
#define MSG_TYPE_PAGE_RESPONSE 5

/* Debug configuration - set to 1 to enable debug, 0 to disable */
#define VMOTIONOS_DEBUG 0

/* Debug printing macros */
#if VMOTIONOS_DEBUG
#define vmotionos_debug(fmt, ...) \
    pr_info("VMotionOS DEBUG: " fmt "\n", ##__VA_ARGS__)
#else
#define vmotionos_debug(fmt, ...) do { } while (0)
#endif

#define vmotionos_info(fmt, ...) \
    pr_info("VMotionOS: " fmt "\n", ##__VA_ARGS__)

#define vmotionos_warn(fmt, ...) \
    pr_warn("VMotionOS: " fmt "\n", ##__VA_ARGS__)

#define vmotionos_err(fmt, ...) \
    pr_err("VMotionOS: " fmt "\n", ##__VA_ARGS__)

/* VMA data structure for process memory recovery */
struct vmotionos_vma {
    unsigned long vm_start;                 /* Start address */
    unsigned long vm_end;                   /* End address */
    unsigned long vm_flags;                 /* VMA flags (r/w/x/s) */
    unsigned long vm_pgoff;                 /* File offset */
    unsigned long vm_page_prot;             /* Page protection bits */
    char vm_file_path[MAX_PATH_LEN];       /* File path if backed by file */
    int has_file;                          /* 1 if file-backed, 0 if anonymous */
} __attribute__((packed));

/* CPU register data structure for x86-64 */
struct vmotionos_regs {
    /* General purpose registers */
    unsigned long r15, r14, r13, r12;       
    unsigned long bp, bx;                   
    unsigned long r11, r10, r9, r8;         
    unsigned long ax, cx, dx, si, di;       
    
    /* Special registers */
    unsigned long ip;                       /* Instruction pointer (RIP) */
    unsigned long sp;                       /* Stack pointer (RSP) */
    unsigned long flags;                    /* CPU flags register */
    unsigned long orig_ax;                  /* Original syscall number */
    
    /* Segment registers */
    unsigned long cs, ss;                   /* Code segment, stack segment */
    
    /* Thread-specific bases (FS/GS) */
    unsigned long fsbase;                   /* FS segment base */
    unsigned long gsbase;                   /* GS segment base */
} __attribute__((packed));

/* Memory management data structure */
struct vmotionos_mm {
    unsigned long start_code;              /* Code segment start */
    unsigned long end_code;                /* Code segment end */
    unsigned long start_data;              /* Data segment start */
    unsigned long end_data;                /* Data segment end */
    unsigned long start_brk;               /* Heap start */
    unsigned long brk;                     /* Current heap end */
    unsigned long start_stack;             /* Stack start */
    unsigned long arg_start;               /* Arguments start */
    unsigned long arg_end;                 /* Arguments end */
    unsigned long env_start;               /* Environment start */
    unsigned long env_end;                 /* Environment end */
    int vma_count;                         /* Number of VMAs */
    struct vmotionos_vma vmas[MAX_VMA_COUNT]; /* VMA array */
} __attribute__((packed));

/* File descriptor data structure */
struct vmotionos_fd {
    int fd_num;                            /* File descriptor number */
    unsigned int f_flags;                  /* File flags */
    unsigned long f_pos;                   /* File position */
    char f_path[MAX_PATH_LEN];            /* File path */
    int is_valid;                          /* 1 if valid FD, 0 if empty slot */
    int close_on_exec;                     /* 1 if close-on-exec set */
} __attribute__((packed));

/* Files table data structure */
struct vmotionos_files {
    unsigned int next_fd;                  /* Next FD to allocate */
    unsigned int max_fds;                  /* Maximum FDs in table */
    int resize_in_progress;                /* Resize operation flag */
    int fd_count;                          /* Number of valid FDs */
    struct vmotionos_fd fds[MAX_FD_COUNT]; /* File descriptor array */
} __attribute__((packed));

/* Complete thread/process data structure */
struct vmotionos_thread {
    int pid;                               /* Process ID */
    char comm[MAX_COMM_LEN];              /* Command name */
    struct vmotionos_regs regs;            /* CPU registers */
    struct vmotionos_mm mm;                /* Memory management structure */
    struct vmotionos_files files;          /* Open files and FD table */
} __attribute__((packed));

/* Page request structure for network page fault handling */
struct vmotionos_page_request {
    int source_pid;                        /* Source process ID */
    unsigned long page_addr;               /* Page address to request */
    unsigned long vma_start;               /* VMA start address */
    unsigned long vma_end;                 /* VMA end address */
    char source_ip[16];                    /* Source node IP address */
    int source_port;                       /* Source node port */
    char dest_ip[16];                      /* Destination node IP address */
    int dest_port;                         /* Destination node port */
} __attribute__((packed));

/* Page response structure for network page fault handling */
struct vmotionos_page_response {
    int source_pid;                        /* Source process ID */
    unsigned long page_addr;               /* Page address */
    unsigned char page_data[PAGE_SIZE];    /* Page data */
    int success;                           /* 1 if successful, 0 if failed */
} __attribute__((packed));

/* Typed message structure for network transmission */
struct vmotionos_message {
    int msg_type;                          /* Message type */
    size_t data_size;                      /* Size of data payload */
    char data[];                           /* Flexible array for data */
} __attribute__((packed));

/* Core function declarations */
int vmotionos_extract_thread_data(pid_t pid, struct vmotionos_thread *thread);
int vmotionos_extract_files_data(struct task_struct *task, struct vmotionos_files *files);
void vmotionos_print_thread_data(struct vmotionos_thread *thread);
void vmotionos_print_files_data(struct vmotionos_files *files);
int vmotionos_send_thread_data(struct vmotionos_thread *thread, char *dest_ip, int dest_port);
void vmotionos_process_received_thread(struct vmotionos_thread *thread);

/* Page fault handling functions */
int vmotionos_send_page_request(struct vmotionos_page_request *request, char *dest_ip, int dest_port);
int vmotionos_send_page_response(struct vmotionos_page_response *response, char *dest_ip, int dest_port);
void vmotionos_handle_page_request(struct vmotionos_page_request *request);
void vmotionos_handle_page_response(struct vmotionos_page_response *response);
void vmotionos_set_page_response(unsigned long page_addr, void *page_data, int success);

#endif /* _VMOTIONOS_H */