#include "vmotionos_client.h"

static struct socket *client_socket = NULL;

/**
 * Send a UDP message to specified destination
 */
int vmotionos_send_message(struct vmotionos_message_params *msg_params)
{
    struct sockaddr_in dest_addr;
    struct msghdr msg;
    struct kvec iov;
    int ret;
    bool socket_created = false;

    if (!msg_params || !msg_params->dest_ip || !msg_params->message) {
        vmotionos_err("Client: Invalid parameters\n");
        return -EINVAL;
    }

    if (msg_params->msg_len == 0) {
        msg_params->msg_len = strlen(msg_params->message);
    }

    vmotionos_debug("Client: Sending %zu bytes to %s:%d\n", 
                   msg_params->msg_len, msg_params->dest_ip, msg_params->dest_port);

    /* Create socket if not already created */
    if (!client_socket) {
        ret = sock_create(AF_INET, SOCK_DGRAM, IPPROTO_UDP, &client_socket);
        if (ret < 0) {
            vmotionos_err("Client: Failed to create socket, error %d\n", ret);
            return ret;
        }
        socket_created = true;
        vmotionos_debug("Client: Created new UDP socket\n");
    }

    /* Setup destination address */
    memset(&dest_addr, 0, sizeof(dest_addr));
    dest_addr.sin_family = AF_INET;
    dest_addr.sin_port = htons(msg_params->dest_port);
    dest_addr.sin_addr.s_addr = in_aton(msg_params->dest_ip);

    /* Setup message structure */
    memset(&msg, 0, sizeof(msg));
    msg.msg_name = &dest_addr;
    msg.msg_namelen = sizeof(dest_addr);

    iov.iov_base = msg_params->message;
    iov.iov_len = msg_params->msg_len;

    /* Send message */
    ret = kernel_sendmsg(client_socket, &msg, &iov, 1, msg_params->msg_len);
    if (ret < 0) {
        vmotionos_err("Client: Failed to send message to %s:%d, error %d\n",
                      msg_params->dest_ip, msg_params->dest_port, ret);
        if (socket_created) {
            sock_release(client_socket);
            client_socket = NULL;
        }
        return ret;
    }

    vmotionos_debug("Client: Successfully sent %d bytes\n", ret);
    return 0;
}

void vmotionos_client_cleanup(void)
{
    if (client_socket) {
        sock_release(client_socket);
        client_socket = NULL;
        vmotionos_debug("Client: Socket released\n");
    }
}
