/*
 * VMotiONOS Core - Custom Page Fault Handler Testing
 * 
 * Simple approach:
 * 1. Create empty child process (no args to copy_process)
 * 2. Manually create mm_struct for child
 * 3. Copy VMA structures (but not pages) from source
 * 4. Set custom page fault handler for each VMA
 * 5. Test page fault interception when child accesses memory
 * 
 * Note: Assumes vmotionos.h defines:
 * - vmotionos_info(), vmotionos_err(), vmotionos_warn() macros
 */

#include <linux/sched.h>
#include <linux/sched/task.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/pid.h>
#include <linux/proc_fs.h>
#include <linux/uaccess.h>
#include "vmotionos.h"

#include <linux/signal.h>
#include <linux/delay.h>
#include <linux/sched/signal.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/posix_types.h>
#include <asm/prctl.h> /* For do_arch_prctl_64 */
#include <asm/processor.h> /* For __USER_CS, __USER_DS */

/* For memory management */
#include <linux/mmu_context.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/sched/mm.h>
#include <linux/vmalloc.h>
#include <linux/rmap.h>

/* For file descriptor management */
#include <linux/fdtable.h>
#include <linux/file.h>
#include <linux/fs_struct.h>
#include <linux/syscalls.h>

/* Additional includes for missing definitions */
#include <linux/mm_types.h>
#include <linux/mman.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/atomic.h>
#include <linux/rwsem.h>
#include <linux/mmap_lock.h>
#include <linux/highmem.h>
#include <linux/pagemap.h>
#include <linux/memcontrol.h>
#include <linux/swap.h>
#include <linux/rmap.h>
#include <linux/page-flags.h>
#include <linux/hugetlb.h>
#include <linux/sched/coredump.h>
#include <linux/printk.h>
#include <linux/refcount.h> /* Add this include at the top */

#include <linux/pgtable.h> // For pgd_offset, pud_offset, etc.
#include <asm/pgtable_types.h> // For pte_pfn, pgd_none, etc.

/* x86-specific includes for register access */
#include <asm/ptrace.h>
#include <linux/sched/task_stack.h>

#define SUSPEND_TIMEOUT_MS 1000
#define SUSPEND_POLL_INTERVAL_MS 10

#define VMOTIONOS_DEBUG 1

static struct proc_dir_entry *proc_entry;

/* Function prototypes */
static int vmotionos_fork_process_thread(void *data);

/* Copy all x86 registers from source task to current task */
static int vmotionos_copy_x86_registers(struct task_struct *source_task)
{
	struct pt_regs *source_regs, *current_regs;
	struct thread_struct *source_thread;
	int cpu;
#if VMOTIONOS_DEBUG
	vmotionos_info(
		"=== Copying x86 registers from source task PID %d ===\n",
		source_task->pid);
#endif

	/* Get source registers */
	source_regs = task_pt_regs(source_task);
	if (!source_regs) {
		vmotionos_err("Failed to get source pt_regs\n");
		return -EFAULT;
	}

	source_thread = &source_task->thread;

	/* Log source registers */
#if VMOTIONOS_DEBUG
	vmotionos_info("Source registers:\n");
	vmotionos_info("  RIP: 0x%016lx  RSP: 0x%016lx  RBP: 0x%016lx\n",
		       source_regs->ip, source_regs->sp, source_regs->bp);
	vmotionos_info("  RAX: 0x%016lx  RBX: 0x%016lx  RCX: 0x%016lx\n",
		       source_regs->ax, source_regs->bx, source_regs->cx);
#endif

	/* ✅ CRITICAL: Get current process's user-space registers */
	current_regs = current_pt_regs();
	if (!current_regs) {
		vmotionos_err("Failed to get current pt_regs\n");
		return -EFAULT;
	}

	/* ✅ CRITICAL: Disable preemption during register manipulation */
	cpu = get_cpu();

	/* Copy general purpose registers individually (like Popcorn) */
	current_regs->r15 = source_regs->r15;
	current_regs->r14 = source_regs->r14;
	current_regs->r13 = source_regs->r13;
	current_regs->r12 = source_regs->r12;
	current_regs->bp = source_regs->bp;
	current_regs->bx = source_regs->bx;
	current_regs->r11 = source_regs->r11;
	current_regs->r10 = source_regs->r10;
	current_regs->r9 = source_regs->r9;
	current_regs->r8 = source_regs->r8;
	current_regs->ax = source_regs->ax;
	current_regs->cx = source_regs->cx;
	current_regs->dx = source_regs->dx;
	current_regs->si = source_regs->si;
	current_regs->di = source_regs->di;
	current_regs->ip = source_regs->ip; /* Instruction pointer */
	current_regs->sp = source_regs->sp; /* Stack pointer */
	current_regs->flags = source_regs->flags; /* CPU flags */

	/* ✅ CRITICAL: Copy orig_ax for proper syscall handling */
	current_regs->orig_ax = source_regs->orig_ax;

	/* ✅ CRITICAL: Set proper user space segments (not copied from source) */
	current_regs->cs = __USER_CS; /* User code segment */
	current_regs->ss = __USER_DS; /* User data/stack segment */

	/* ✅ CRITICAL: Handle FS/GS bases properly using arch_prctl */
#ifdef CONFIG_X86_64
	if (source_thread->fsbase) {
#if VMOTIONOS_DEBUG
		vmotionos_info("Setting FS base: 0x%016lx\n",
			       source_thread->fsbase);
#endif
		do_arch_prctl_64(current, ARCH_SET_FS, source_thread->fsbase);
	}

	if (source_thread->gsbase) {
#if VMOTIONOS_DEBUG
		vmotionos_info("Setting GS base: 0x%016lx\n",
			       source_thread->gsbase);
#endif
		do_arch_prctl_64(current, ARCH_SET_GS, source_thread->gsbase);
	}
#endif
#if VMOTIONOS_DEBUG
	vmotionos_info("=== REGISTER VALIDATION & CLEANING ===\n");
#endif

	/* Lock the memory map before VMA access */
	down_read(&current->mm->mmap_lock);

	/* Check if RIP is in valid executable memory */
	struct vm_area_struct *ip_vma = find_vma(current->mm, current_regs->ip);
	if (!ip_vma || current_regs->ip < ip_vma->vm_start ||
	    !(ip_vma->vm_flags & VM_EXEC)) {
		vmotionos_err("❌ INVALID RIP: 0x%lx not in executable VMA!\n",
			      current_regs->ip);
		up_read(&current->mm->mmap_lock);
		return -EFAULT;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ RIP 0x%lx is in valid executable VMA\n",
		       current_regs->ip);
#endif

	/* Check if RSP is in valid stack memory */
	struct vm_area_struct *sp_vma = find_vma(current->mm, current_regs->sp);
	if (!sp_vma || current_regs->sp < sp_vma->vm_start ||
	    !(sp_vma->vm_flags & VM_GROWSDOWN)) {
		vmotionos_err("❌ INVALID RSP: 0x%lx not in stack VMA!\n",
			      current_regs->sp);
		up_read(&current->mm->mmap_lock);
		return -EFAULT;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ RSP 0x%lx is in valid stack VMA\n",
		       current_regs->sp);
#endif

	/* Unlock the memory map */
	up_read(&current->mm->mmap_lock);

	/* Clean potential syscall error state */
	if ((long)current_regs->ax < 0 && (long)current_regs->ax >= -4095) {
#if VMOTIONOS_DEBUG
		vmotionos_info("🧹 Cleaning syscall error: RAX=%ld -> 0\n",
			       (long)current_regs->ax);
#endif
		current_regs->ax = 0;
	}

	/* Ensure clean CPU flags */
	current_regs->flags |= 0x200; /* Enable interrupts */
	current_regs->flags &= ~0x1; /* Clear carry flag */

#if VMOTIONOS_DEBUG
	vmotionos_info("=== REGISTER VALIDATION COMPLETED ===\n");
#endif

	/* Re-enable preemption */
	put_cpu();

	/* Log final registers */
#if VMOTIONOS_DEBUG
	vmotionos_info("Registers copied successfully:\n");
	vmotionos_info("  RIP: 0x%016lx  RSP: 0x%016lx  RBP: 0x%016lx\n",
		       current_regs->ip, current_regs->sp, current_regs->bp);
	vmotionos_info("  CS: 0x%04x  SS: 0x%04x  FLAGS: 0x%016lx\n",
		       (unsigned int)current_regs->cs,
		       (unsigned int)current_regs->ss, current_regs->flags);

	vmotionos_info("=== Register copy completed successfully ===\n");
#endif
	return 0;
}

/* Store source task information for fault handler */
struct vmotionos_vma_data {
	struct task_struct *source_task;
	unsigned long source_start; /* Original VMA start in source */
	unsigned long source_end; /* Original VMA end in source */
	struct file *source_file; /* Original file backing (if any) */
	unsigned long source_pgoff; /* Original page offset */
	vm_flags_t source_flags; /* Original VM flags */
	unsigned long source_pfn; /* Source PFN for direct mapping */
	pgprot_t pgprot;
};

// Helper to get physical address by walking source page tables
static resource_size_t get_phys_addr_from_pt(struct mm_struct *mm,
					     unsigned long addr)
{
	pgd_t *pgd;
	p4d_t *p4d;
	pud_t *pud;
	pmd_t *pmd;
	pte_t *pte;
	resource_size_t phys = 0;

	mmap_read_lock(mm); // Safe lock for reading page tables

	pgd = pgd_offset(mm, addr);
	if (pgd_none(*pgd) || pgd_bad(*pgd))
		goto out;

	p4d = p4d_offset(pgd, addr);
	if (p4d_none(*p4d) || p4d_bad(*p4d))
		goto out;

	pud = pud_offset(p4d, addr);
	if (pud_none(*pud) || pud_bad(*pud))
		goto out;

	pmd = pmd_offset(pud, addr);
	if (pmd_none(*pmd) || pmd_bad(*pmd))
		goto out;

	pte = pte_offset_map(pmd, addr);
	if (!pte || pte_none(*pte)) {
		pte_unmap(pte);
		goto out;
	}

	phys = (resource_size_t)pte_pfn(*pte) << PAGE_SHIFT;
	pte_unmap(pte);

out:
	mmap_read_unlock(mm);
	return phys;
}

static vm_fault_t vmotionos_vm_fault(struct vm_fault *vmf)
{
	struct vm_area_struct *vma = vmf->vma;
	unsigned long address = vmf->address;
	unsigned long faulted_masked_address;
	struct vmotionos_vma_data *vma_data;
	struct page *new_page;
	void *child_addr;

	/* Page-align the address */
	faulted_masked_address = address & PAGE_MASK;

#if VMOTIONOS_DEBUG
	vmotionos_info("*** PAGE FAULT CAUGHT! ***\n");
	vmotionos_info("  VMA: 0x%lx-0x%lx\n", vma->vm_start, vma->vm_end);
	vmotionos_info("  Fault address: 0x%lx (page: 0x%lx)\n", address,
		       faulted_masked_address);
	vmotionos_info("  Fault flags: 0x%x\n", vmf->flags);
	vmotionos_info("  PID: %d (%s)\n", current->pid, current->comm);
#endif

#if VMOTIONOS_DEBUG
	if (vmf->flags & FAULT_FLAG_WRITE) {
		vmotionos_info("  -> WRITE fault\n");
	} else {
		vmotionos_info("  -> READ fault\n");
	}
#endif

#if VMOTIONOS_DEBUG
	if (vma->vm_flags & VM_GROWSDOWN) {
		vmotionos_info("  -> STACK fault at 0x%lx\n",
			       faulted_masked_address);
		vmotionos_info("  -> Current RSP: 0x%lx\n",
			       task_pt_regs(current)->sp);
		vmotionos_info("  -> Stack page vs RSP offset: %ld bytes\n",
			       (long)task_pt_regs(current)->sp -
				       (long)faulted_masked_address);
	}
#endif

#if VMOTIONOS_DEBUG
	if (vma->vm_flags & VM_EXEC) {
		vmotionos_info("  -> CODE segment fault\n");
	} else if (vma->vm_flags & VM_GROWSDOWN) {
		vmotionos_info("  -> STACK segment fault\n");
	} else if (vma->vm_flags & VM_WRITE) {
		vmotionos_info("  -> DATA segment fault\n");
	}
#endif

	/* Get source task information from VMA private data */
	vma_data = (struct vmotionos_vma_data *)vma->vm_private_data;
	if (!vma_data || !vma_data->source_task) {
		vmotionos_err(
			"No source task information in VMA private data\n");
		return VM_FAULT_SIGBUS;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("  -> Copying page from source PID %d\n",
		       vma_data->source_task->pid);
	vmotionos_info("  -> Original VMA: 0x%lx-0x%lx, flags: 0x%lx\n",
		       vma_data->source_start, vma_data->source_end,
		       vma_data->source_flags);

	/* Log VMA type information */
	if (vma->vm_file) {
		vmotionos_info("  -> File-backed VMA (pgoff: 0x%lx)\n",
			       vma_data->source_pgoff);
	} else {
		vmotionos_info("  -> Anonymous VMA\n");
	}
#endif
	/* Check if we're accessing within original bounds */
	if (faulted_masked_address < vma_data->source_start ||
	    faulted_masked_address >= vma_data->source_end) {
		vmotionos_err(
			"Page fault outside original VMA bounds: 0x%lx not in [0x%lx-0x%lx]\n",
			faulted_masked_address, vma_data->source_start,
			vma_data->source_end);
		return VM_FAULT_SIGBUS;
	}

	/* Allocate new page for child */
	new_page = alloc_page(GFP_HIGHUSER_MOVABLE | __GFP_ZERO);
	if (!new_page) {
		vmotionos_err("Failed to allocate new page for child\n");
		return VM_FAULT_OOM;
	}

	child_addr = kmap(new_page);

	/* Try normal copy first */
	if (access_process_vm(vma_data->source_task, faulted_masked_address,
			      child_addr, PAGE_SIZE, 0) == PAGE_SIZE) {
#if VMOTIONOS_DEBUG
		vmotionos_info("  -> Normal copy successful");
#endif
	} else {
		/* Fallback to page table walk for special regions (e.g., VDSO/PFNMAP) */
		resource_size_t phys_addr = get_phys_addr_from_pt(
			vma_data->source_task->mm, faulted_masked_address);
		if (phys_addr) {
			void *src = __va(phys_addr); // Kernel direct mapping
			memcpy(child_addr, src, PAGE_SIZE);
#if VMOTIONOS_DEBUG
			vmotionos_info(
				"  -> PT walk copy successful for special region (phys: 0x%llx)",
				phys_addr);
#endif
		} else {
			vmotionos_err(
				"Failed to get phys addr via PT walk for source\n");
			kunmap(new_page);
			__free_page(new_page);
			return VM_FAULT_SIGBUS;
		}
	}

	kunmap(new_page);
#if VMOTIONOS_DEBUG
	vmotionos_info("Page successfully copied from source to child\n");
#endif

	lock_page(new_page);
	vmf->page = new_page;

#if VMOTIONOS_DEBUG
	vmotionos_info(
		"Page successfully remapped with original pgprot (0x%lx)\n",
		pgprot_val(vma_data->pgprot));
#endif
	return VM_FAULT_LOCKED;
}

/* Custom VM operations with our page fault handler */
static const struct vm_operations_struct vmotionos_vm_ops = {
	.fault = vmotionos_vm_fault,
};

/* Copy VMA structure completely (no memory mapping yet) */
static int vmotionos_copy_vma_structure(struct mm_struct *child_mm,
					struct vm_area_struct *src_vma,
					struct task_struct *source_task)
{
	struct vm_area_struct *new_vma;
	unsigned long len;
	struct vmotionos_vma_data *vma_data;
	vm_flags_t flags_src;

	len = src_vma->vm_end - src_vma->vm_start;
#if VMOTIONOS_DEBUG
	vmotionos_info("Copying VMA structure: 0x%lx-0x%lx (%lu bytes) (%s)\n",
		       src_vma->vm_start, src_vma->vm_end, len,
		       (src_vma->vm_flags & VM_EXEC)	  ? "CODE" :
		       (src_vma->vm_flags & VM_GROWSDOWN) ? "STACK" :
		       (src_vma->vm_flags & VM_WRITE)	  ? "DATA" :
							    "OTHER");
#endif
	/* Allocate new VMA structure */
	new_vma = vm_area_alloc(child_mm);
	if (!new_vma) {
		vmotionos_err("Failed to allocate VMA structure\n");
		return -ENOMEM;
	}

	/* Allocate VMA data to store source information */
	vma_data = kmalloc(sizeof(struct vmotionos_vma_data), GFP_KERNEL);
	if (!vma_data) {
		vmotionos_err("Failed to allocate VMA data\n");
		vm_area_free(new_vma);
		return -ENOMEM;
	}

	/* Store complete source VMA information */
	vma_data->source_task = source_task;
	vma_data->source_start = src_vma->vm_start;
	vma_data->source_end = src_vma->vm_end;
	vma_data->pgprot = src_vma->vm_page_prot;
	vma_data->source_file = src_vma->vm_file;
	vma_data->source_pgoff = src_vma->vm_pgoff;
	vma_data->source_flags = src_vma->vm_flags;
	vma_data->source_pfn = src_vma->vm_start >> PAGE_SHIFT;

	/* Copy all VMA fields from source */
	new_vma->vm_start = src_vma->vm_start;
	new_vma->vm_end = src_vma->vm_end;
	new_vma->vm_mm = child_mm;

	/* Copy flags but modify for migration */
	flags_src = src_vma->vm_flags;
#if VMOTIONOS_DEBUG
	vmotionos_debug("Source VMA flags: 0x%lx\n", flags_src);
#endif
	vm_flags_set(new_vma, flags_src);
#if VMOTIONOS_DEBUG
	vmotionos_debug("New VMA flags: 0x%lx\n", new_vma->vm_flags);
#endif

	/* Copy protection and offset */
	new_vma->vm_page_prot = src_vma->vm_page_prot;
	new_vma->vm_pgoff = src_vma->vm_pgoff;

	/* Handle file backing */
	if (src_vma->vm_file) {
		/* For file-backed VMAs, we'll handle file access through fault handler */
		new_vma->vm_file =
			get_file(src_vma->vm_file); // Increment file reference
#if VMOTIONOS_DEBUG
		vmotionos_info(
			"  -> File-backed VMA, will handle through fault handler\n");
#endif
	} else {
		new_vma->vm_file = NULL;
	}

	/* Set our private data and custom VM operations */
	new_vma->vm_private_data = vma_data;
	new_vma->vm_ops = &vmotionos_vm_ops;

	/* Copy anon_vma information if present (for anonymous VMAs) */
	if (src_vma->anon_vma) {
		/* Don't copy anon_vma directly - we'll handle through fault handler */
		new_vma->anon_vma = NULL;
#if VMOTIONOS_DEBUG
		vmotionos_info(
			"  -> Anonymous VMA, will handle through fault handler\n");
#endif
	}

	/* Insert VMA into child's memory map */
	if (insert_vm_struct(child_mm, new_vma)) {
		vmotionos_err("Failed to insert VMA into child mm\n");
		kfree(vma_data);
		vm_area_free(new_vma);
		return -ENOMEM;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info(
		"  -> VMA structure copied completely with custom fault handler\n");
	vmotionos_info("     Original flags: 0x%lx -> New flags: 0x%lx\n",
		       src_vma->vm_flags, new_vma->vm_flags);
#endif
	return 0;
}

/* Copy VMAs from source to child (structures only) */
static int
vmotionos_copy_vmas_with_fault_handler(struct task_struct *source_task,
				       struct task_struct *child_task)
{
	struct mm_struct *source_mm, *child_mm;
	struct vm_area_struct *vma;
	struct vma_iterator vmi;
	int ret = 0;
	int vma_count = 0;
#if VMOTIONOS_DEBUG
	vmotionos_info("Copying VMA structures from source to child...\n");
#endif
	source_mm = source_task->mm;
	if (!source_mm) {
		vmotionos_err(
			"Source task has no memory management structure\n");
		return -EFAULT;
	}

	child_mm = child_task->mm;
	if (!child_mm) {
		vmotionos_err(
			"Child task has no memory management structure\n");
		return -EFAULT;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("Source MM info:\n");
	vmotionos_info("  Start code: 0x%lx, End code: 0x%lx\n",
		       source_mm->start_code, source_mm->end_code);
	vmotionos_info("  Start data: 0x%lx, End data: 0x%lx\n",
		       source_mm->start_data, source_mm->end_data);
	vmotionos_info("  Start stack: 0x%lx\n", source_mm->start_stack);
#endif
	/* Copy basic mm_struct fields to child */
	child_mm->start_code = source_mm->start_code;
	child_mm->end_code = source_mm->end_code;
	child_mm->start_data = source_mm->start_data;
	child_mm->end_data = source_mm->end_data;
	child_mm->start_brk = source_mm->start_brk;
	child_mm->brk = source_mm->brk;
	child_mm->start_stack = source_mm->start_stack;
	child_mm->arg_start = source_mm->arg_start;
	child_mm->arg_end = source_mm->arg_end;
	child_mm->env_start = source_mm->env_start;
	child_mm->env_end = source_mm->env_end;

	/* Lock both memory spaces */
	down_read(&source_mm->mmap_lock);
	down_write(&child_mm->mmap_lock);

	/* Iterate through all VMAs in source process */
	vma_iter_init(&vmi, source_mm, 0);
	for_each_vma(vmi, vma) {
		ret = vmotionos_copy_vma_structure(child_mm, vma, source_task);
		if (ret < 0) {
			vmotionos_err("Failed to copy VMA 0x%lx-0x%lx: %d\n",
				      vma->vm_start, vma->vm_end, ret);
			break;
		}
		vma_count++;
	}

	/* Unlock memory spaces */
	up_write(&child_mm->mmap_lock);
	up_read(&source_mm->mmap_lock);
#if VMOTIONOS_DEBUG
	if (ret == 0) {
		vmotionos_info("VMA structure copy completed successfully\n");
		vmotionos_info("  Total VMAs copied: %d\n", vma_count);
		vmotionos_info("  All VMAs have custom page fault handlers\n");
		vmotionos_info("  No actual memory pages allocated yet\n");
	}
#endif
	return ret;
}

static int vmotionos_suspend_process(struct task_struct *task, pid_t pid)
{
	long waited = 0;
	long syscall_wait = 0;
	bool stopped = false;
	const long MAX_SYSCALL_WAIT_MS =
		2000; /* Wait up to 2 seconds for syscall completion */
#if VMOTIONOS_DEBUG
	vmotionos_info("Suspending process PID %d...\n", pid);

	/* ✅ STEP 1: Wait for any ongoing system calls to complete */
	vmotionos_info("Checking if process is in system call...\n");
#endif
	while (syscall_wait < MAX_SYSCALL_WAIT_MS) {
		unsigned long state = task->__state;
#if VMOTIONOS_DEBUG
		vmotionos_info(
			"Process state: %lu (%s)\n", state,
			(state == TASK_RUNNING)		? "RUNNING" :
			(state == TASK_INTERRUPTIBLE)	? "INTERRUPTIBLE" :
			(state == TASK_UNINTERRUPTIBLE) ? "UNINTERRUPTIBLE" :
			(state == TASK_STOPPED)		? "STOPPED" :
			(state == TASK_TRACED)		? "TRACED" :
							  "OTHER");
#endif
		/* If process is in uninterruptible sleep (deep in syscall), wait */
		if (state == TASK_UNINTERRUPTIBLE) {
#if VMOTIONOS_DEBUG
			vmotionos_info(
				"Process in UNINTERRUPTIBLE state (syscall), waiting...\n");
#endif
			msleep(50); /* Wait 50ms for syscall to complete */
			syscall_wait += 50;
			continue;
		}

		/* If process is in interruptible sleep, it might be in a syscall too */
		if (state == TASK_INTERRUPTIBLE) {
#if VMOTIONOS_DEBUG
			vmotionos_info(
				"Process in INTERRUPTIBLE state, checking if safe to suspend...\n");
#endif
			msleep(20); /* Brief wait to let any quick syscalls finish */
			syscall_wait += 20;

			/* Check again after brief wait */
			if (task->__state == TASK_INTERRUPTIBLE) {
#if VMOTIONOS_DEBUG
				vmotionos_info(
					"Process still INTERRUPTIBLE after wait, should be safe to suspend\n");
#endif
				break;
			}
			continue;
		}

		/* If process is running or stopped, it's safe to suspend */
		if (state == TASK_RUNNING || state == TASK_STOPPED) {
#if VMOTIONOS_DEBUG
			vmotionos_info(
				"Process in safe state (%s) for suspension\n",
				(state == TASK_RUNNING) ? "RUNNING" :
							  "STOPPED");
#endif
			break;
		}

		/* For any other state, wait a bit */
		msleep(20);
		syscall_wait += 20;
	}

	if (syscall_wait >= MAX_SYSCALL_WAIT_MS) {
		vmotionos_warn(
			"Timeout waiting for syscall completion, proceeding anyway\n");
	} else {
		vmotionos_info("Safe to suspend after waiting %ld ms\n",
			       syscall_wait);
	}

	/* ✅ STEP 2: Now send SIGSTOP */
#if VMOTIONOS_DEBUG
	vmotionos_info("Sending SIGSTOP to process PID %d\n", pid);
#endif
	send_sig(SIGSTOP, task, 0);

	/* ✅ STEP 3: Wait for process to actually stop */
	while (waited < SUSPEND_TIMEOUT_MS) {
		if (task->__state == TASK_STOPPED) {
			stopped = true;
			break;
		}
		msleep(SUSPEND_POLL_INTERVAL_MS);
		waited += SUSPEND_POLL_INTERVAL_MS;
	}

	if (!stopped) {
		vmotionos_err(
			"Process PID %d did not enter TASK_STOPPED state within timeout!\n",
			pid);
		return -EAGAIN;
	} else {
		vmotionos_info(
			"Process PID %d is now suspended (TASK_STOPPED) after %ld ms total\n",
			pid, syscall_wait + waited);
	}

	return 0;
}
static int vmotionos_copy_filesystem_context(struct task_struct *source_task)
{
	struct fs_struct *source_fs;
	struct fs_struct *new_fs;

	return 0;

	source_fs = source_task->fs;
	if (!source_fs) {
		vmotionos_err("Source task has no fs_struct\n");
		return -EFAULT;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("Copying filesystem context from source task PID %d\n",
		       source_task->pid);
#endif
	/* Copy the filesystem structure */
	new_fs = copy_fs_struct(source_fs);
	if (!new_fs) {
		vmotionos_err("Failed to copy fs_struct\n");
		return -ENOMEM;
	}

	/* Log the filesystem context being copied */
#if VMOTIONOS_DEBUG
	vmotionos_info("Filesystem context copied:\n");
	vmotionos_info("  umask: 0%o\n", new_fs->umask);
	vmotionos_info("  users: %d\n", new_fs->users);
	vmotionos_info("  in_exec: %d\n", new_fs->in_exec);
#endif
	/* Replace current fs_struct */
	task_lock(current);
	if (current->fs) {
		free_fs_struct(current->fs);
	}
	current->fs = new_fs;
	task_unlock(current);
#if VMOTIONOS_DEBUG
	vmotionos_info("Filesystem context copied successfully\n");
	vmotionos_info("Enhanced filesystem copying includes:\n");
	vmotionos_info("  ✅ Root directory path\n");
	vmotionos_info("  ✅ Current working directory path\n");
	vmotionos_info("  ✅ File creation mask (umask)\n");
	vmotionos_info("  ✅ Execution state (in_exec)\n");
	vmotionos_info("  ✅ Reference count (users)\n");
#endif
	return 0;
}
static int vmotionos_copy_file_descriptors(struct task_struct *source_task)
{
	struct files_struct *source_files;
	struct files_struct *new_files;
	struct fdtable *source_fdt, *new_fdt;

	source_files = source_task->files;
	if (!source_files) {
		vmotionos_err("Source task has no files_struct\n");
		return -EFAULT;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("Copying file descriptors from source task PID %d\n",
		       source_task->pid);
#endif
	/* Get reference to source files_struct */
	new_files = dup_fd(source_files, NULL);
	if (IS_ERR(new_files)) {
		vmotionos_err("Failed to duplicate files_struct: %ld\n",
			      PTR_ERR(new_files));
		return PTR_ERR(new_files);
	}

	/* Now copy the additional fields that dup_fd() doesn't handle properly */
	spin_lock(&source_files->file_lock);
	spin_lock(&new_files->file_lock);

	source_fdt = files_fdtable(source_files);
	new_fdt = files_fdtable(new_files);

	/* Log current state before copying additional fields */
#if VMOTIONOS_DEBUG
	vmotionos_info("Source files_struct state:\n");
	vmotionos_info("  next_fd: %u\n", source_files->next_fd);
	vmotionos_info("  max_fds: %u\n", source_fdt->max_fds);
	vmotionos_info("  resize_in_progress: %s\n",
		       source_files->resize_in_progress ? "true" : "false");
	vmotionos_info("  count: %d\n", atomic_read(&source_files->count));

	vmotionos_info("New files_struct state (after dup_fd):\n");
	vmotionos_info("  next_fd: %u (should be copied from source)\n",
		       new_files->next_fd);
	vmotionos_info("  max_fds: %u (should match source if larger)\n",
		       new_fdt->max_fds);
	vmotionos_info(
		"  resize_in_progress: %s (should be copied from source)\n",
		new_files->resize_in_progress ? "true" : "false");
	vmotionos_info("  count: %d\n", atomic_read(&new_files->count));
#endif
	/* Fix 1: Copy the next_fd field that dup_fd() resets to 0 */
	new_files->next_fd = source_files->next_fd;

	/* Fix 2: Copy resize_in_progress state */
	new_files->resize_in_progress = source_files->resize_in_progress;

	/* Fix 3: Handle max_fds properly - dup_fd() already handles this correctly 
	  * by expanding the table if needed, but let's log if there's a mismatch */
#if VMOTIONOS_DEBUG
	if (new_fdt->max_fds != source_fdt->max_fds) {
		vmotionos_info("Note: max_fds differs - Source: %u, New: %u\n",
			       source_fdt->max_fds, new_fdt->max_fds);
		vmotionos_info(
			"This is normal if dup_fd() optimized the table size\n");
	}

	vmotionos_info("After copying additional fields:\n");
	vmotionos_info("  next_fd: %u\n", new_files->next_fd);
	vmotionos_info("  max_fds: %u\n", new_fdt->max_fds);
	vmotionos_info("  resize_in_progress: %s\n",
		       new_files->resize_in_progress ? "true" : "false");
#endif
	spin_unlock(&new_files->file_lock);
	spin_unlock(&source_files->file_lock);

	/* Replace current files_struct */
	task_lock(current);
	if (current->files) {
		put_files_struct(current->files);
	}
	current->files = new_files;
	task_unlock(current);
#if VMOTIONOS_DEBUG
	vmotionos_info("File descriptors copied successfully\n");
	vmotionos_info("Enhanced copying includes:\n");
	vmotionos_info(
		"  ✅ File pointers (fd array) with proper reference counting\n");
	vmotionos_info("  ✅ Open file descriptors bitmap (open_fds)\n");
	vmotionos_info("  ✅ Close-on-exec bitmap (close_on_exec)\n");
	vmotionos_info("  ✅ Full FDs bitmap (full_fds_bits)\n");
	vmotionos_info("  ✅ Maximum FDs (max_fds) - handled by dup_fd()\n");
	vmotionos_info("  ✅ Next FD allocation hint (next_fd) - FIXED!\n");
	vmotionos_info(
		"  ✅ Resize in progress state (resize_in_progress) - FIXED!\n");
	vmotionos_info("  ✅ Reference count (count)\n");
	vmotionos_info("  ✅ File lock initialization\n");
	vmotionos_info("  ✅ Wait queue initialization (resize_wait)\n");

	/* Additional validation - check that file descriptors are properly copied */
	vmotionos_info("Validating file descriptor copying:\n");
#endif
	spin_lock(&new_files->file_lock);
	new_fdt = files_fdtable(new_files);

	int fd_count = 0;
	int max_check = min_t(unsigned int, new_fdt->max_fds,
			      32); /* Check first 32 FDs */

	for (int i = 0; i < max_check; i++) {
		if (test_bit(i, new_fdt->open_fds)) {
			struct file *file = new_fdt->fd[i];
			if (file) {
				fd_count++;
#if VMOTIONOS_DEBUG
				vmotionos_info(
					"  FD %d: file=%p, f_flags=0x%x\n", i,
					file, file->f_flags);
#endif
			}
		}
	}

	spin_unlock(&new_files->file_lock);
#if VMOTIONOS_DEBUG
	vmotionos_info("Total validated FDs: %d\n", fd_count);
#endif
	return 0;
}
static int vmotionos_copy_signal_handlers(struct task_struct *source_task)
{
	struct sighand_struct *source_sighand, *new_sighand;
	int i;

	return 0;
#if VMOTIONOS_DEBUG
	vmotionos_info("Copying signal handlers from source PID %d\n",
		       source_task->pid);
#endif
	source_sighand = source_task->sighand;
	if (!source_sighand) {
		vmotionos_err("Source task has no sighand_struct\n");
		return -EFAULT;
	}

	/* Allocate new signal handler structure */
	new_sighand = kmem_cache_alloc(sighand_cachep, GFP_KERNEL);
	if (!new_sighand) {
		vmotionos_err("Failed to allocate sighand_struct\n");
		return -ENOMEM;
	}

	/* Initialize the new sighand structure */
	refcount_set(&new_sighand->count, 1); /* ✅ FIXED: Use refcount_set */
	spin_lock_init(&new_sighand->siglock);

	/* Copy all signal handlers with proper locking */
	spin_lock_irq(&source_sighand->siglock);

	for (i = 0; i < _NSIG; i++) {
		new_sighand->action[i] = source_sighand->action[i];

		/* Log important signal handlers for debugging */
		if (source_sighand->action[i].sa.sa_handler != SIG_DFL &&
		    source_sighand->action[i].sa.sa_handler != SIG_IGN) {
#if VMOTIONOS_DEBUG
			vmotionos_info(
				"  Signal %d: handler=0x%lx, flags=0x%lx\n",
				i + 1,
				(unsigned long)source_sighand->action[i]
					.sa.sa_handler,
				(unsigned long)source_sighand->action[i]
					.sa.sa_flags);
#endif
		}
	}

	spin_unlock_irq(&source_sighand->siglock);

	/* Replace current signal handlers */
	task_lock(current);

	/* ✅ FIXED: Use proper cleanup function */
	if (current->sighand) {
		__cleanup_sighand(
			current->sighand); /* Use kernel cleanup function */
	}

	current->sighand = new_sighand;
	task_unlock(current);

#if VMOTIONOS_DEBUG
	vmotionos_info("Signal handlers copied: %d actions copied\n", _NSIG);
#endif
	return 0;
}
static int vmotionos_copy_signal_disposition(struct task_struct *source_task)
{
#if VMOTIONOS_DEBUG
	vmotionos_info("Copying signal disposition from source PID %d\n",
		       source_task->pid);
#endif

	return 0;
	/* Copy signal masks */
	spin_lock_irq(&source_task->sighand->siglock);

	current->blocked = source_task->blocked;
	current->real_blocked = source_task->real_blocked;
	current->saved_sigmask = source_task->saved_sigmask;

	/* Log signal masks for debugging */
#if VMOTIONOS_DEBUG
	vmotionos_info("  Blocked signals: 0x%lx\n", current->blocked.sig[0]);
	vmotionos_info("  Real blocked: 0x%lx\n", current->real_blocked.sig[0]);
#endif
	spin_unlock_irq(&source_task->sighand->siglock);

	/* Copy other signal-related fields */
	current->sas_ss_sp = source_task->sas_ss_sp;
	current->sas_ss_size = source_task->sas_ss_size;
	current->sas_ss_flags = source_task->sas_ss_flags;
#if VMOTIONOS_DEBUG
	vmotionos_info("Signal disposition copied successfully\n");
#endif
	return 0;
}

static int vmotionos_fork_process_thread(void *data)
{
	pid_t source_pid = *(pid_t *)data;
	struct pid *pid_struct;
	struct task_struct *source_task;
	struct mm_struct *new_mm;
	int ret;

	kfree(data); // Free the allocated memory

	/* Check if process exists */
	rcu_read_lock();
	pid_struct = find_get_pid(source_pid);
	if (!pid_struct) {
		rcu_read_unlock();
		vmotionos_err("ERROR: Process PID %d not found!\n", source_pid);
		return -ESRCH;
	}

	source_task = get_pid_task(pid_struct, PIDTYPE_PID);
	rcu_read_unlock();

	if (!source_task) {
		put_pid(pid_struct);
		vmotionos_err("ERROR: Could not get task for PID %d\n",
			      source_pid);
		return -ESRCH;
	}

	/* Suspend source process */
	ret = vmotionos_suspend_process(source_task, source_pid);
	if (ret < 0) {
		put_task_struct(source_task);
		put_pid(pid_struct);
		return ret;
	}

	/* CRITICAL: Reparent to init process (PID 1) */
	struct task_struct *init_process;

	rcu_read_lock();
	init_process = find_task_by_vpid(1); // Find PID 1
	if (init_process) {
		get_task_struct(init_process); // Get reference
	}
	rcu_read_unlock();

	if (!init_process) {
		vmotionos_err("Could not find init process!\n");
		put_task_struct(source_task);
		put_pid(pid_struct);
		return -ESRCH;
	}

	write_lock_irq(&tasklist_lock);

	/* Remove from current parent */
	list_del_init(&current->sibling);

	/* Set init as parent */
	current->real_parent = init_process;
	current->parent = init_process;

	/* Add to init's children */
	list_add_tail(&current->sibling, &init_process->children);

	write_unlock_irq(&tasklist_lock);

	put_task_struct(init_process); // Release reference
#if VMOTIONOS_DEBUG
	vmotionos_info("Reparented PID %d to init (PID 1)\n", current->pid);
#endif
	/* CRITICAL: Remove PF_KTHREAD and get clean MM */
	current->flags &= ~PF_KTHREAD;

	/* Allocate fresh mm_struct */
	new_mm = mm_alloc();
	if (!new_mm) {
		vmotionos_err("Failed to allocate mm_struct\n");
		put_task_struct(source_task);
		put_pid(pid_struct);
		return -ENOMEM;
	}

	/* Replace use_mm(new_mm) with: */
	task_lock(current);
	if (current->mm) {
		mmput(current->mm); /* Release old mm if any */
	}
	current->mm = new_mm;
	current->active_mm = new_mm;
	atomic_inc(&new_mm->mm_users);
	task_unlock(current);
#if VMOTIONOS_DEBUG
	vmotionos_info("=== MIGRATION STARTING - FOLLOWING KERNEL ORDER ===\n");
#endif
	/* ===== FOLLOW KERNEL'S copy_process ORDER ===== */

	/* 1. 📁 copy_files() - File descriptors FIRST */
#if VMOTIONOS_DEBUG
	vmotionos_info("Step 1: Copying file descriptors...\n");
#endif
	ret = vmotionos_copy_file_descriptors(source_task);
	if (ret < 0) {
		vmotionos_err("Failed to copy file descriptors: %d\n", ret);
		goto cleanup;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ File descriptors copied successfully\n");
#endif
	/* 2. 🗂️ copy_fs() - Filesystem context */
#if VMOTIONOS_DEBUG
	vmotionos_info("Step 2: Copying filesystem context...\n");
#endif
	ret = vmotionos_copy_filesystem_context(source_task);
	if (ret < 0) {
		vmotionos_err("Failed to copy filesystem context: %d\n", ret);
		goto cleanup;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ Filesystem context copied successfully\n");
#endif

	/* 3. 🛡️ copy_sighand() - Signal handlers */
#if VMOTIONOS_DEBUG
	vmotionos_info("Step 3: Copying signal handlers...\n");
#endif
	ret = vmotionos_copy_signal_handlers(source_task);
	if (ret < 0) {
		vmotionos_err("Failed to copy signal handlers: %d\n", ret);
		goto cleanup;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ Signal handlers copied successfully\n");
#endif
	/* 4. 📡 copy_signal() - Signal disposition */
#if VMOTIONOS_DEBUG
	vmotionos_info("Step 4: Copying signal disposition...\n");
#endif
	ret = vmotionos_copy_signal_disposition(source_task);
	if (ret < 0) {
		vmotionos_err("Failed to copy signal disposition: %d\n", ret);
		goto cleanup;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ Signal disposition copied successfully\n");
#endif
	/* 5. 🧠 copy_mm() - Memory management */
#if VMOTIONOS_DEBUG
	vmotionos_info("Step 5: Copying memory management...\n");
#endif
	ret = vmotionos_copy_vmas_with_fault_handler(source_task, current);
	if (ret < 0) {
		vmotionos_err("Failed to copy VMAs: %d\n", ret);
		goto cleanup;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ Memory management copied successfully\n");
#endif
	/* 6. 🔧 copy_thread() - Thread state (registers) LAST */
#if VMOTIONOS_DEBUG
	vmotionos_info("Step 6: Copying thread state (registers)...\n");
#endif
	ret = vmotionos_copy_x86_registers(source_task);
	if (ret < 0) {
		vmotionos_err("Failed to copy registers: %d\n", ret);
		goto cleanup;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ Thread state copied successfully\n");
#endif
	strscpy(current->comm, source_task->comm);
#if VMOTIONOS_DEBUG
	vmotionos_info("✅ Process name copied: %s\n", current->comm);

	vmotionos_info(
		"🎉 MIGRATION COMPLETED SUCCESSFULLY IN KERNEL ORDER! 🎉\n");
	vmotionos_info("Process should resume execution normally...\n");

	//print full analysis of the process
	vmotionos_info(
		"Printing full analysis of the DESTINATION PROCESS...\n");
	vmotionos_analyze_process_selective(current, VMOTIONOS_PRINT_ALL);
#endif
	/* Cleanup */
	put_task_struct(source_task);
	put_pid(pid_struct);
	return 0;

cleanup:
	put_task_struct(source_task);
	put_pid(pid_struct);
	return ret;
}

int vmotionos_fork_process(pid_t source_pid)
{
	pid_t *pid_data;

#if VMOTIONOS_DEBUG
	vmotionos_info("Starting fork process for pid %d\n", source_pid);
	vmotionos_info(
		"*******************************************************\n");

	vmotionos_info("Starting fork process for pid %d\n", source_pid);
#endif
	// Allocate memory for PID that won't go out of scope
	pid_data = kmalloc(sizeof(pid_t), GFP_KERNEL);
	if (!pid_data) {
		return -ENOMEM;
	}
	*pid_data = source_pid;

	// find task_struct of source_pid
	struct task_struct *source_task = find_task_by_vpid(source_pid);
	if (!source_task) {
		vmotionos_err("ERROR: Could not get task for PID %d\n",
			      source_pid);
		return -ESRCH;
	}
#if VMOTIONOS_DEBUG
	vmotionos_info("Printing full analysis of the SOURCE PROCESS...\n");
	vmotionos_analyze_process_selective(source_task, VMOTIONOS_PRINT_ALL);
#endif

	while (kernel_thread(vmotionos_fork_process_thread, pid_data, "user",
			     SIGCHLD) < 0) {
		schedule();
	}
	return 0;
}

/* /proc file write function */
static ssize_t proc_write(struct file *file, const char __user *buffer,
			  size_t count, loff_t *pos)
{
	char buf[32];
	pid_t pid;
	int result;

	vmotionos_info(
		"/proc/vmotionos_copy: Received write request (%zu bytes)\n",
		count);

	if (count >= sizeof(buf)) {
		vmotionos_err("Input too long (max %zu bytes)\n",
			      sizeof(buf) - 1);
		return -EINVAL;
	}

	if (copy_from_user(buf, buffer, count)) {
		vmotionos_err("Failed to copy data from user space\n");
		return -EFAULT;
	}

	buf[count] = '\0';
	vmotionos_info("Received string: '%s'\n", buf);

	if (kstrtoint(buf, 10, &pid) != 0) {
		vmotionos_err("Invalid PID format: '%s'\n", buf);
		return -EINVAL;
	}

	vmotionos_info("Parsed PID: %d\n", pid);

	if (pid <= 0) {
		vmotionos_err("Invalid PID value: %d (must be > 0)\n", pid);
		return -EINVAL;
	}

	/* Call our copy function */
	// result = vmotionos_copy_process(pid);
	result = vmotionos_fork_process(pid);
	if (result < 0) {
		vmotionos_err("Process copy failed with error: %d\n", result);
		return result;
	}

	vmotionos_info("/proc write operation completed successfully\n");
	return count;
}

/* /proc file operations */
static const struct proc_ops proc_fops = {
	.proc_write = proc_write,
};

/* VMotiONOS initialization */
static int __init vmotionos_init(void)
{
	vmotionos_info("===========================================\n");
	vmotionos_info("  VMotiONOS Page Fault Handler Test v1.0  \n");
	vmotionos_info("===========================================\n");
	vmotionos_info("VMotiONOS initializing...\n");

	/* Create /proc/vmotionos_copy entry */
	proc_entry = proc_create("vmotionos_copy", 0200, NULL, &proc_fops);
	if (!proc_entry) {
		vmotionos_err("FAILED to create /proc/vmotionos_copy\n");
		return -ENOMEM;
	}

	vmotionos_info("Created /proc/vmotionos_copy (write-only)\n");
	vmotionos_info("VMotiONOS initialization complete!\n");
	vmotionos_info("===========================================\n");
	vmotionos_info("USAGE: echo <PID> > /proc/vmotionos_copy\n");
	vmotionos_info("Child will fault and our handler will print!\n");
	vmotionos_info("===========================================\n");

	return 0;
}

/* Built-in subsystem initialization */
subsys_initcall(vmotionos_init);