#include "vmotionos.h"
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <asm/ptrace.h>
#include <linux/export.h>
#include <linux/sched/task_stack.h>
#include <linux/fs.h>
#include <linux/file.h>
#include <linux/fdtable.h>
#include <linux/path.h>
#include <linux/dcache.h>
#include <linux/sched/signal.h>
#include <linux/mm.h>
#include <linux/mmap_lock.h>
#include <linux/printk.h>
#include <linux/string.h>
#include <linux/mm_types.h>
#include <linux/cpumask.h>

/* Analysis flags - can be combined using bitwise OR */
#define VMOTIONOS_PRINT_REGISTERS    (1 << 0)  /* 0x01 */
#define VMOTIONOS_PRINT_MEMORY       (1 << 1)  /* 0x02 */
#define VMOTIONOS_PRINT_SIGNALS      (1 << 2)  /* 0x04 */
#define VMOTIONOS_PRINT_FILES        (1 << 3)  /* 0x08 */
#define VMOTIONOS_PRINT_BASIC        (1 << 4)  /* 0x10 */
#define VMOTIONOS_PRINT_THREAD       (1 << 5)  /* 0x20 */
#define VMOTIONOS_PRINT_NAMESPACE    (1 << 6)  /* 0x40 */
#define VMOTIONOS_PRINT_CGROUP       (1 << 7)  /* 0x80 */

/* Convenience combinations */
#define VMOTIONOS_PRINT_ALL          (0xFF)
#define VMOTIONOS_PRINT_ESSENTIAL    (VMOTIONOS_PRINT_BASIC | VMOTIONOS_PRINT_REGISTERS | VMOTIONOS_PRINT_MEMORY)
#define VMOTIONOS_PRINT_MINIMAL      (VMOTIONOS_PRINT_BASIC)
#define VMOTIONOS_PRINT_DEBUG        (VMOTIONOS_PRINT_REGISTERS | VMOTIONOS_PRINT_MEMORY | VMOTIONOS_PRINT_SIGNALS)

static void vmotionos_print_basic_info(struct task_struct *task)
{
    if (!task) {
        vmotionos_info("Invalid task for basic analysis\n");
        return;
    }
    
    vmotionos_info("\n=== BASIC PROCESS INFORMATION ===\n");
    vmotionos_info("Process ID (PID):      %d\n", task->pid);
    vmotionos_info("Parent PID (PPID):     %d\n", task->parent ? task->parent->pid : 0);
    vmotionos_info("Thread Group ID:       %d\n", task->tgid);
    vmotionos_info("Process Name:          %s\n", task->comm);
    vmotionos_info("Process State:         %u (%s)\n", task->__state, 
                   task->__state == TASK_RUNNING ? "RUNNING" :
                   task->__state == TASK_INTERRUPTIBLE ? "SLEEPING" :
                   task->__state == TASK_UNINTERRUPTIBLE ? "UNINTERRUPTIBLE" :
                   task->__state == TASK_STOPPED ? "STOPPED" :
                   task->__state == TASK_TRACED ? "TRACED" : "OTHER");
    vmotionos_info("Exit State:            %d\n", task->exit_state);
    vmotionos_info("Exit Code:             %d\n", task->exit_code);
    vmotionos_info("Process Flags:         0x%08x\n", task->flags);
    vmotionos_info("Real UID:              %d\n", task->cred->uid.val);
    vmotionos_info("Effective UID:         %d\n", task->cred->euid.val);
    vmotionos_info("Real GID:              %d\n", task->cred->gid.val);
    vmotionos_info("Effective GID:         %d\n", task->cred->egid.val);
    vmotionos_info("Priority:              %d\n", task->prio);
    vmotionos_info("Nice Value:            %d\n", task->static_prio - 120);
    vmotionos_info("Policy:                %d\n", task->policy);
    vmotionos_info("CPU Number:            %d\n", task_cpu(task));
}

static void vmotionos_print_registers(struct task_struct *task)
{
    struct pt_regs *regs;
    
    if (!task) {
        vmotionos_info("No stack information available for task %d\n", task ? task->pid : -1);
        return;
    }
    
    regs = task_pt_regs(task);
    if (!regs) {
        vmotionos_info("No register information available for task %d\n", task->pid);
        return;
    }
    
    vmotionos_info("\n=== CPU REGISTERS (x86_64) ===\n");
    
    vmotionos_info("General Purpose Registers:\n");
    vmotionos_info("  RAX: 0x%016lx  RBX: 0x%016lx  RCX: 0x%016lx\n", 
                   regs->ax, regs->bx, regs->cx);
    vmotionos_info("  RDX: 0x%016lx  RSI: 0x%016lx  RDI: 0x%016lx\n", 
                   regs->dx, regs->si, regs->di);
    vmotionos_info("  R8 : 0x%016lx  R9 : 0x%016lx  R10: 0x%016lx\n", 
                   regs->r8, regs->r9, regs->r10);
    vmotionos_info("  R11: 0x%016lx  R12: 0x%016lx  R13: 0x%016lx\n", 
                   regs->r11, regs->r12, regs->r13);
    vmotionos_info("  R14: 0x%016lx  R15: 0x%016lx\n", 
                   regs->r14, regs->r15);
    
    vmotionos_info("Stack & Frame Pointers:\n");
    vmotionos_info("  RSP: 0x%016lx  RBP: 0x%016lx\n", 
                   regs->sp, regs->bp);
    
    vmotionos_info("Instruction Pointer & Flags:\n");
    vmotionos_info("  RIP: 0x%016lx  FLAGS: 0x%016lx\n", 
                   regs->ip, regs->flags);
    
    vmotionos_info("Segment Registers:\n");
    vmotionos_info("  CS: 0x%04x  SS: 0x%04x\n", 
                   (unsigned int)(regs->cs & 0xFFFF), (unsigned int)(regs->ss & 0xFFFF));
    
    vmotionos_info("Original RAX (syscall): 0x%016lx\n", regs->orig_ax);
    
    /* Print flag breakdown */
    vmotionos_info("CPU Flags: CF:%d PF:%d AF:%d ZF:%d SF:%d TF:%d IF:%d DF:%d OF:%d\n",
                   !!(regs->flags & (1 << 0)),   /* CF - Carry Flag */
                   !!(regs->flags & (1 << 2)),   /* PF - Parity Flag */
                   !!(regs->flags & (1 << 4)),   /* AF - Auxiliary Flag */
                   !!(regs->flags & (1 << 6)),   /* ZF - Zero Flag */
                   !!(regs->flags & (1 << 7)),   /* SF - Sign Flag */
                   !!(regs->flags & (1 << 8)),   /* TF - Trap Flag */
                   !!(regs->flags & (1 << 9)),   /* IF - Interrupt Flag */
                   !!(regs->flags & (1 << 10)),  /* DF - Direction Flag */
                   !!(regs->flags & (1 << 11))); /* OF - Overflow Flag */
    vmotionos_info("Extended: IOPL:%d NT:%d RF:%d VM:%d AC:%d VIF:%d VIP:%d ID:%d\n",
                   (int)((regs->flags >> 12) & 3), /* IOPL - I/O Privilege Level */
                   !!(regs->flags & (1 << 14)),    /* NT - Nested Task */
                   !!(regs->flags & (1 << 16)),    /* RF - Resume Flag */
                   !!(regs->flags & (1 << 17)),    /* VM - Virtual 8086 Mode */
                   !!(regs->flags & (1 << 18)),    /* AC - Alignment Check */
                   !!(regs->flags & (1 << 19)),    /* VIF - Virtual Interrupt Flag */
                   !!(regs->flags & (1 << 20)),    /* VIP - Virtual Interrupt Pending */
                   !!(regs->flags & (1 << 21)));   /* ID - ID Flag */
}

static void vmotionos_print_signals(struct task_struct *task)
{
    unsigned long flags;
    sigset_t pending, shpending, blocked, ignored, caught;
    int i;
    
    if (!task) {
        vmotionos_info("Invalid task for signal analysis\n");
        return;
    }
    
    vmotionos_info("\n=== SIGNAL INFORMATION ===\n");
    vmotionos_info("Signal information for PID %d (%s):\n", task->pid, task->comm);
    
    /* Initialize signal sets */
    sigemptyset(&pending);
    sigemptyset(&shpending);
    sigemptyset(&blocked);
    sigemptyset(&ignored);
    sigemptyset(&caught);
    
    /* Lock signal handler and get signal information */
    if (lock_task_sighand(task, &flags)) {
        pending = task->pending.signal;
        shpending = task->signal->shared_pending.signal;
        blocked = task->blocked;
        
        /* Collect ignored and caught signals */
        for (i = 1; i <= _NSIG; i++) {
            struct k_sigaction *ka = &task->sighand->action[i-1];
            if (ka->sa.sa_handler == SIG_IGN)
                sigaddset(&ignored, i);
            else if (ka->sa.sa_handler != SIG_DFL)
                sigaddset(&caught, i);
        }
        
        vmotionos_info("\nSignal Masks (hex format):\n");
        vmotionos_info("  Pending signals:       0x%016lx\n", pending.sig[0]);
        vmotionos_info("  Shared pending:        0x%016lx\n", shpending.sig[0]);
        vmotionos_info("  Blocked signals:       0x%016lx\n", blocked.sig[0]);
        vmotionos_info("  Ignored signals:       0x%016lx\n", ignored.sig[0]);
        vmotionos_info("  Caught signals:        0x%016lx\n", caught.sig[0]);
        
        /* Print detailed signal information */
        vmotionos_info("\nDetailed Signal Analysis:\n");
        
        /* Show pending signals */
        vmotionos_info("  Pending signals: ");
        bool has_pending = false;
        for (i = 1; i <= _NSIG && i <= 64; i++) {
            if (sigismember(&pending, i)) {
                vmotionos_info("%d ", i);
                has_pending = true;
            }
        }
        if (!has_pending) vmotionos_info("none");
        vmotionos_info("\n");
        
        /* Show blocked signals */
        vmotionos_info("  Blocked signals: ");
        bool has_blocked = false;
        for (i = 1; i <= _NSIG && i <= 64; i++) {
            if (sigismember(&blocked, i)) {
                vmotionos_info("%d ", i);
                has_blocked = true;
            }
        }
        if (!has_blocked) vmotionos_info("none");
        vmotionos_info("\n");

vmotionos_info("\n=== DETAILED SIGNAL HANDLERS ===\n");
vmotionos_info("+------+----------+-------------+----------------------------------------+\n");
vmotionos_info("| Sig  | Handler  | Flags       | Mask                                   |\n");
vmotionos_info("+------+----------+-------------+----------------------------------------+\n");

for (i = 1; i <= _NSIG; i++) {
    struct k_sigaction *ka = &task->sighand->action[i-1];
    char handler_str[32];
    
    if (ka->sa.sa_handler == SIG_DFL) {
        strcpy(handler_str, "DFL");
    } else if (ka->sa.sa_handler == SIG_IGN) {
        strcpy(handler_str, "IGN");
    } else {
        snprintf(handler_str, sizeof(handler_str), "%p", ka->sa.sa_handler);
    }
    
    // Print mask as hex for simplicity (first 64 bits)
    unsigned long mask = ka->sa.sa_mask.sig[0];
    
    vmotionos_info("| %4d | %-8s | 0x%08lx  | 0x%016lx                     |\n",
                   i, handler_str, ka->sa.sa_flags, mask);
    
    // Print flag breakdown
    char flags_str[256] = {0};
    char *fp = flags_str;
    int flen = sizeof(flags_str);
    if (ka->sa.sa_flags & SA_NOCLDSTOP) { snprintf(fp, flen, "NOCLDSTOP "); fp += strlen(fp); flen -= strlen(fp); }
    if (ka->sa.sa_flags & SA_NOCLDWAIT) { snprintf(fp, flen, "NOCLDWAIT "); fp += strlen(fp); flen -= strlen(fp); }
    if (ka->sa.sa_flags & SA_SIGINFO) { snprintf(fp, flen, "SIGINFO "); fp += strlen(fp); flen -= strlen(fp); }
    if (ka->sa.sa_flags & SA_ONSTACK) { snprintf(fp, flen, "ONSTACK "); fp += strlen(fp); flen -= strlen(fp); }
    if (ka->sa.sa_flags & SA_RESTART) { snprintf(fp, flen, "RESTART "); fp += strlen(fp); flen -= strlen(fp); }
    if (ka->sa.sa_flags & SA_NODEFER) { snprintf(fp, flen, "NODEFER "); fp += strlen(fp); flen -= strlen(fp); }
    if (ka->sa.sa_flags & SA_RESETHAND) { snprintf(fp, flen, "RESETHAND "); fp += strlen(fp); flen -= strlen(fp); }
    if (ka->sa.sa_flags & SA_RESTORER) { snprintf(fp, flen, "RESTORER "); fp += strlen(fp); flen -= strlen(fp); }
    vmotionos_info("   Flags: %s\n", flags_str);
    
    // If mask is non-zero, list set signals
    if (mask != 0) {
        char mask_str[512] = {0};
        char *mp = mask_str;
        int mlen = sizeof(mask_str);
        bool has_mask = false;
        int j;
        for (j = 1; j <= 64; j++) {
            if (mask & (1UL << (j-1))) {
                snprintf(mp, mlen, "%d ", j);
                mp += strlen(mp);
                mlen -= strlen(mp);
                has_mask = true;
            }
        }
        if (!has_mask) snprintf(mp, mlen, "none");
        vmotionos_info("   Mask signals: %s\n", mask_str);
    }
}

vmotionos_info("+------+----------+-------------+----------------------------------------+\n");

        unlock_task_sighand(task, &flags);
    } else {
        vmotionos_info("Could not lock signal handler for task %d\n", task->pid);
    }
}

static void vmotionos_print_file_descriptors(struct task_struct *task)
{
    struct files_struct *files;
    struct fdtable *fdt;
    int i, fd_count = 0;
    
    if (!task) {
        vmotionos_info("Invalid task for FD analysis\n");
        return;
    }
    
    vmotionos_info("\n=== FILE DESCRIPTORS ===\n");
    vmotionos_info("Open file descriptors for PID %d (%s):\n", task->pid, task->comm);
    
    /* Get files structure */
    task_lock(task);
    files = task->files;
    if (files)
        atomic_inc(&files->count);
    task_unlock(task);
    
    if (!files) {
        vmotionos_info("No file descriptor table available\n");
        return;
    }
    
    /* Lock the files structure */
    spin_lock(&files->file_lock);
    fdt = files_fdtable(files);
    
    vmotionos_info("\nFile Descriptor Table Information:\n");
    vmotionos_info("  Max FDs:           %u\n", fdt->max_fds);
    vmotionos_info("  Next FD:           %u\n", files->next_fd);
    
    vmotionos_info("\n=== OPEN FILE DESCRIPTORS ===\n");
    vmotionos_info("+-----+----------+-------------------+----------------------------------------+\n");
    vmotionos_info("| FD  | Flags    | Inode             | Path/Description                       |\n");
    vmotionos_info("+-----+----------+-------------------+----------------------------------------+\n");
    
    /* Iterate through file descriptors */
    for (i = 0; i < fdt->max_fds && i < 32; i++) { /* Limit to first 32 FDs */
        struct file *file;
        
        if (!test_bit(i, fdt->open_fds))
            continue;
            
        file = fdt->fd[i];
        if (!file)
            continue;
            
        /* Get file information */
        const char *path_name = "unknown";
        unsigned long inode_num = 0;
        unsigned int flags = 0;
        char path_buffer[128];
        
        if (file->f_path.dentry && file->f_path.dentry->d_name.name) {
            path_name = file->f_path.dentry->d_name.name;
        }
        
        if (file->f_inode) {
            inode_num = file->f_inode->i_ino;
        }
        
        flags = file->f_flags;
        
        /* Try to get more complete path information */
        if (file->f_path.dentry && file->f_path.mnt) {
            char *full_path = d_path(&file->f_path, path_buffer, sizeof(path_buffer));
            if (!IS_ERR(full_path)) {
                path_name = full_path;
            }
        }
        
        vmotionos_info("| %3d | 0x%06x | %8lu          | %-38s |\n",
                      i, flags, inode_num, path_name);
        
        fd_count++;
    }
    
    vmotionos_info("+-----+----------+-------------------+----------------------------------------+\n");
    vmotionos_info("Total open file descriptors shown: %d\n", fd_count);
    
    spin_unlock(&files->file_lock);
    put_files_struct(files);
}

static void vmotionos_print_vma_flags(vm_flags_t flags)
{
    vmotionos_info("0x%08lx (", flags);
    
    /* Basic access flags */
    if (flags & VM_READ) vmotionos_info("READ ");
    if (flags & VM_WRITE) vmotionos_info("WRITE ");
    if (flags & VM_EXEC) vmotionos_info("EXEC ");
    if (flags & VM_SHARED) vmotionos_info("SHARED ");
    
    /* mprotect() limits */
    if (flags & VM_MAYREAD) vmotionos_info("MAYREAD ");
    if (flags & VM_MAYWRITE) vmotionos_info("MAYWRITE ");
    if (flags & VM_MAYEXEC) vmotionos_info("MAYEXEC ");
    if (flags & VM_MAYSHARE) vmotionos_info("MAYSHARE ");
    
    /* General segment info */
    if (flags & VM_GROWSDOWN) vmotionos_info("GROWSDOWN ");
    
    /* UFFD flags */
#ifdef CONFIG_MMU
    if (flags & VM_UFFD_MISSING) vmotionos_info("UFFD_MISSING ");
#else
    if (flags & VM_MAYOVERLAY) vmotionos_info("MAYOVERLAY ");
#endif
    if (flags & VM_UFFD_WP) vmotionos_info("UFFD_WP ");
    
    /* Memory management flags */
    if (flags & VM_PFNMAP) vmotionos_info("PFNMAP ");
    if (flags & VM_LOCKED) vmotionos_info("LOCKED ");
    if (flags & VM_IO) vmotionos_info("IO ");
    
    /* Access pattern hints (madvise) */
    if (flags & VM_SEQ_READ) vmotionos_info("SEQ_READ ");
    if (flags & VM_RAND_READ) vmotionos_info("RAND_READ ");
    
    /* Fork and memory management behavior */
    if (flags & VM_DONTCOPY) vmotionos_info("DONTCOPY ");
    if (flags & VM_DONTEXPAND) vmotionos_info("DONTEXPAND ");
    if (flags & VM_LOCKONFAULT) vmotionos_info("LOCKONFAULT ");
    if (flags & VM_ACCOUNT) vmotionos_info("ACCOUNT ");
    if (flags & VM_NORESERVE) vmotionos_info("NORESERVE ");
    
    /* Huge pages */
    if (flags & VM_HUGETLB) vmotionos_info("HUGETLB ");
    if (flags & VM_HUGEPAGE) vmotionos_info("HUGEPAGE ");
    if (flags & VM_NOHUGEPAGE) vmotionos_info("NOHUGEPAGE ");
    
    /* Sync and architecture specific */
    if (flags & VM_SYNC) vmotionos_info("SYNC ");
    if (flags & VM_ARCH_1) vmotionos_info("ARCH_1 ");
    
    /* Security and debugging */
    if (flags & VM_WIPEONFORK) vmotionos_info("WIPEONFORK ");
    if (flags & VM_DONTDUMP) vmotionos_info("DONTDUMP ");
    
    /* Soft dirty tracking */
#ifdef CONFIG_MEM_SOFT_DIRTY
    if (flags & VM_SOFTDIRTY) vmotionos_info("SOFTDIRTY ");
#endif
    
    /* Mixed mapping and KSM */
    if (flags & VM_MIXEDMAP) vmotionos_info("MIXEDMAP ");
    if (flags & VM_MERGEABLE) vmotionos_info("MERGEABLE ");
    
    vmotionos_info(")");
}

static void vmotionos_print_memory_info(struct task_struct *task)
{
    struct mm_struct *mm;
    struct vm_area_struct *vma;
    int vma_count = 0;
    unsigned long total_size = 0;
    
    if (!task) {
        vmotionos_info("Invalid task for memory analysis\n");
        return;
    }
    
    /* Get memory management struct */
    mm = task->mm;
    if (!mm) {
        vmotionos_info("No memory management information available for task %d\n", task->pid);
        return;
    }
    
    vmotionos_info("\n=== MEMORY INFORMATION ===\n");
    vmotionos_info("Memory management for PID %d (%s):\n", task->pid, task->comm);
    
    /* Print basic memory layout info */
    vmotionos_info("\nMemory Layout:\n");
    vmotionos_info("  Code segment:  0x%016lx - 0x%016lx (%lu KB)\n", 
                   mm->start_code, mm->end_code,
                   (mm->end_code - mm->start_code) >> 10);
    vmotionos_info("  Data segment:  0x%016lx - 0x%016lx (%lu KB)\n", 
                   mm->start_data, mm->end_data,
                   (mm->end_data - mm->start_data) >> 10);
    vmotionos_info("  Heap start:    0x%016lx\n", mm->start_brk);
    vmotionos_info("  Current brk:   0x%016lx (%lu KB heap)\n", 
                   mm->brk, (mm->brk - mm->start_brk) >> 10);
    vmotionos_info("  Stack start:   0x%016lx\n", mm->start_stack);
    
    /* Print memory statistics */
    vmotionos_info("\nMemory Statistics:\n");
    vmotionos_info("  Total VM pages:    %lu (%lu KB)\n", 
                   mm->total_vm, mm->total_vm << (PAGE_SHIFT - 10));
    vmotionos_info("  Map count:         %d\n", mm->map_count);
    
    /* Print VMAs with flags - no truncation */
    vmotionos_info("\n=== VIRTUAL MEMORY AREAS (All VMAs) ===\n");
    
    /* Lock the memory map for reading */
    mmap_read_lock(mm);
    
    vmotionos_info("+-------------------+-------------------+---------+------+---------------+----------------------------------------+\n");
    vmotionos_info("| Start Address     | End Address       | Size KB | Perm | Type          | Description                            |\n");
    vmotionos_info("+-------------------+-------------------+---------+------+---------------+----------------------------------------+\n");
    
    /* Iterate through all VMAs using find_vma */
    vma = find_vma(mm, 0);
    while (vma) {
        unsigned long size = vma->vm_end - vma->vm_start;
        char file_description[64];
        char vma_type[16];
        char perm_str[8];
        char full_path[128];
        
        /* Build permission string */
        perm_str[0] = (vma->vm_flags & VM_READ) ? 'r' : '-';
        perm_str[1] = (vma->vm_flags & VM_WRITE) ? 'w' : '-';
        perm_str[2] = (vma->vm_flags & VM_EXEC) ? 'x' : '-';
        perm_str[3] = (vma->vm_flags & VM_SHARED) ? 's' : 'p';
        perm_str[4] = '\0';
        
        /* Determine VMA type and description */
        if (vma->vm_file) {
            struct dentry *dentry = vma->vm_file->f_path.dentry;
            
            /* Try to get full path */
            char *path_ptr = d_path(&vma->vm_file->f_path, full_path, sizeof(full_path));
            if (IS_ERR(path_ptr)) {
                if (dentry && dentry->d_name.name) {
                    snprintf(file_description, sizeof(file_description), "%.60s", 
                            dentry->d_name.name);
                } else {
                    strcpy(file_description, "[unknown file]");
                }
            } else {
                snprintf(file_description, sizeof(file_description), "%.60s", path_ptr);
            }
            
            /* Determine file-backed VMA type */
            if (vma->vm_flags & VM_EXEC) {
                if (vma->vm_flags & VM_WRITE) {
                    strcpy(vma_type, "FILE_RWX");
                } else {
                    strcpy(vma_type, "FILE_RX");
                }
            } else if (vma->vm_flags & VM_WRITE) {
                strcpy(vma_type, "FILE_RW");
            } else {
                strcpy(vma_type, "FILE_RO");
            }
        } else {
            /* Anonymous VMA - determine type based on characteristics */
            if (vma->vm_start <= mm->start_brk && vma->vm_end >= mm->brk) {
                strcpy(vma_type, "HEAP");
                snprintf(file_description, sizeof(file_description), 
                        "Process heap (brk: 0x%lx)", mm->brk);
            } else if (vma->vm_flags & VM_GROWSDOWN) {
                strcpy(vma_type, "STACK");
                snprintf(file_description, sizeof(file_description), 
                        "Thread stack (grows down)");
            } else if (vma->vm_start >= mm->start_code && vma->vm_end <= mm->end_code) {
                strcpy(vma_type, "CODE");
                snprintf(file_description, sizeof(file_description), 
                        "Code segment (0x%lx-0x%lx)", mm->start_code, mm->end_code);
            } else if (vma->vm_start >= mm->start_data && vma->vm_end <= mm->end_data) {
                strcpy(vma_type, "DATA");
                snprintf(file_description, sizeof(file_description), 
                        "Data segment (0x%lx-0x%lx)", mm->start_data, mm->end_data);
            } else if (vma->vm_flags & VM_HUGETLB) {
                strcpy(vma_type, "HUGETLB");
                strcpy(file_description, "Huge page anonymous mapping");
            } else if (vma->vm_flags & VM_PFNMAP) {
                strcpy(vma_type, "PFNMAP");
                strcpy(file_description, "Device/physical memory mapping");
            } else if (vma->vm_flags & VM_IO) {
                strcpy(vma_type, "IO");
                strcpy(file_description, "I/O memory mapping");
            } else if (vma->vm_flags & VM_SHARED) {
                strcpy(vma_type, "ANON_SHARED");
                strcpy(file_description, "Anonymous shared memory");
            } else {
                strcpy(vma_type, "ANON_PRIVATE");
                strcpy(file_description, "Anonymous private memory");
            }
        }
        
        vmotionos_info("| 0x%015lx | 0x%015lx | %7lu | %4s | %-13s | %-38s |\n",
                      vma->vm_start, vma->vm_end, size >> 10, perm_str, vma_type, file_description);
        
        /* Print detailed VMA flags */
        vmotionos_info("  VMA Flags: ");
        vmotionos_print_vma_flags(vma->vm_flags);
        vmotionos_info("\n");
        
        total_size += size;
        vma_count++;
        
        /* Move to next VMA */
        vma = find_vma(mm, vma->vm_end);
    }
    
    vmotionos_info("+-------------------+-------------------+---------+------+---------------+----------------------------------------+\n");
    
    /* Unlock the memory map */
    mmap_read_unlock(mm);
    
    vmotionos_info("Total VMAs: %d, Total mapped size: %lu KB\n", 
                   vma_count, total_size >> 10);
}

static void vmotionos_print_thread_info(struct task_struct *task)
{
    if (!task) {
        vmotionos_info("Invalid task for thread analysis\n");
        return;
    }
    
    vmotionos_info("\n=== THREAD INFORMATION ===\n");
    vmotionos_info("Thread information for PID %d (%s):\n", task->pid, task->comm);
    
    vmotionos_info("Thread Stack:\n");
    vmotionos_info("  Stack pointer:     0x%016lx\n", task->thread.sp);
    vmotionos_info("  Stack start:       0x%016lx\n", (unsigned long)task_stack_page(task));
    vmotionos_info("  Stack size:        %lu bytes\n", THREAD_SIZE);
    
#ifdef CONFIG_X86_64
    vmotionos_info("Segment Registers:\n");
    vmotionos_info("  FS base:           0x%016lx\n", task->thread.fsbase);
    vmotionos_info("  GS base:           0x%016lx\n", task->thread.gsbase);
    vmotionos_info("  FS index:          0x%04x\n", task->thread.fsindex);
    vmotionos_info("  GS index:          0x%04x\n", task->thread.gsindex);
    vmotionos_info("  DS:                0x%04x\n", task->thread.ds);
    vmotionos_info("  ES:                0x%04x\n", task->thread.es);
#endif
    
    vmotionos_info("CPU Context:\n");
    vmotionos_info("  CPU number:        %d\n", task_cpu(task));
    vmotionos_info("  CPU allowed mask:  0x%lx\n", task->cpus_ptr ? cpumask_bits(task->cpus_ptr)[0] : 0);
    vmotionos_info("  On CPU:            %s\n", task_curr(task) ? "yes" : "no");
    vmotionos_info("  Wake CPU:          %d\n", task->wake_cpu);
}

/* Main analysis function with flags */
int vmotionos_analyze_process_selective(struct task_struct *task, unsigned int flags)
{
    if (!task) {
        vmotionos_info("Invalid task provided for analysis\n");
        return -EINVAL;
    }
    
    vmotionos_info("\n========================================\n");
    vmotionos_info("SELECTIVE PROCESS ANALYSIS PID %d (%s)\n", task->pid, task->comm);
    vmotionos_info("Analysis flags: 0x%02x\n", flags);
    vmotionos_info("========================================\n");
    
    /* Print basic info if requested */
    if (flags & VMOTIONOS_PRINT_BASIC) {
        vmotionos_print_basic_info(task);
    }
    
    /* Print CPU registers if requested */
    if (flags & VMOTIONOS_PRINT_REGISTERS) {
        vmotionos_print_registers(task);
    }
    
    /* Print memory information if requested */
    if (flags & VMOTIONOS_PRINT_MEMORY) {
        vmotionos_print_memory_info(task);
    }
    
    /* Print signal information if requested */
    if (flags & VMOTIONOS_PRINT_SIGNALS) {
        vmotionos_print_signals(task);
    }

    /* Print file descriptors if requested */
    if (flags & VMOTIONOS_PRINT_FILES) {
        vmotionos_print_file_descriptors(task);
    }
    
    /* Print thread information if requested */
    if (flags & VMOTIONOS_PRINT_THREAD) {
        vmotionos_print_thread_info(task);
    }
    
    vmotionos_info("\n========================================\n");
    vmotionos_info("Selective analysis completed for PID %d\n", task->pid);
    vmotionos_info("========================================\n");
    
    return 0;
}

/* Convenience functions for common combinations */
int vmotionos_analyze_process_minimal(struct task_struct *task)
{
    return vmotionos_analyze_process_selective(task, VMOTIONOS_PRINT_MINIMAL);
}

int vmotionos_analyze_process_essential(struct task_struct *task)
{
    return vmotionos_analyze_process_selective(task, VMOTIONOS_PRINT_ESSENTIAL);
}

int vmotionos_analyze_process_debug(struct task_struct *task)
{
    return vmotionos_analyze_process_selective(task, VMOTIONOS_PRINT_DEBUG);
}

int vmotionos_analyze_process_full(struct task_struct *task)
{
    return vmotionos_analyze_process_selective(task, VMOTIONOS_PRINT_ALL);
}

/* Export functions */
EXPORT_SYMBOL_GPL(vmotionos_analyze_process_selective);
EXPORT_SYMBOL_GPL(vmotionos_analyze_process_minimal);
EXPORT_SYMBOL_GPL(vmotionos_analyze_process_essential);
EXPORT_SYMBOL_GPL(vmotionos_analyze_process_debug);
EXPORT_SYMBOL_GPL(vmotionos_analyze_process_full);
